package com.imcode.util;

import org.apache.commons.collections.map.MultiValueMap;
import org.apache.commons.collections.MultiMap;
import org.apache.commons.collections.iterators.IteratorEnumeration;
import org.apache.commons.fileupload.*;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.lang.UnhandledException;
import org.apache.oro.text.perl.Perl5Util;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import javax.activation.DataSource;
import java.io.*;
import java.util.*;

/**
 * Extends HttpServletRequest for RFC 1867 form based file upload support from Jakarta Commons-FileUpload
 */
public class MultipartHttpServletRequest extends HttpServletRequestWrapper {

    static final String DEFAULT_CHARSET = "ISO-8859-1";

    private MultiMap /*<FieldName, Collection<FileItem>>*/ fileItemMap;

    public MultipartHttpServletRequest(HttpServletRequest request) throws IOException {
        this(request, -1);
    }

    /**
     * Constructor used to specify a size limit for uploads.
     *
     * @param sizeLimit The maximum size for an upload, in bytes.
     */
    public MultipartHttpServletRequest(HttpServletRequest request, long sizeLimit) throws IOException {
        super(request);
        if (ServletFileUpload.isMultipartContent(request)) {
            ServletFileUpload fileUpload = new ServletFileUpload(new DiskFileItemFactory());
            fileUpload.setSizeMax(sizeLimit);
            fileUpload.setHeaderEncoding("UTF-8");
            List<FileItem> fileItems;
            try {
                fileItems = (List<FileItem>) fileUpload.parseRequest(request);
            } catch (FileUploadException e) {
                throw new IOException(e.getMessage());
            }
            fileItemMap = new MultiValueMap();
            for (FileItem fileItem : fileItems) {
                fileItemMap.put(fileItem.getFieldName(), new BaseNameFileItem(fileItem));
            }
        }
    }

    @Override
    public String getParameter(String key) {
        String[] parameterValues = getParameterValues(key);
        if (null != parameterValues && parameterValues.length > 0) {
            return parameterValues[0];
        }
        return null;
    }

    public DataSourceFileItem getParameterFileItem(String key) {
        DataSourceFileItem[] parameterValues = getParameterFileItems(key);
        if (null != parameterValues && parameterValues.length > 0) {
            return parameterValues[0];
        }
        return null;
    }

    public DataSourceFileItem[] getParameterFileItems(String key) {
        if (null == fileItemMap) {
            return null;
        }
        @SuppressWarnings("unchecked")
        final Collection<DataSourceFileItem> parameterFileItems = (Collection<DataSourceFileItem>) fileItemMap.get(key);
        if (null == parameterFileItems) {
            return null;
        }
        return parameterFileItems.toArray(new DataSourceFileItem[parameterFileItems.size()]);
    }

    @Override
    public Map getParameterMap() {
        Map map = super.getParameterMap();
        if (null != fileItemMap) {
            map = new HashMap(map);
            @SuppressWarnings("unchecked")
            Set<String> fileItemKeys = (Set<String>) fileItemMap.keySet();
            for (String key : fileItemKeys) {
                map.put(key, getParameterValues(key));
            }
        }
        return map;
    }

    @Override
    public Enumeration getParameterNames() {
        @SuppressWarnings("unchecked")
        Enumeration<String> superParameterNames = (Enumeration<String>) super.getParameterNames();
        Set<String> parameterNames = new HashSet<String>();
        while (superParameterNames.hasMoreElements()) {
            parameterNames.add(superParameterNames.nextElement());
        }
        if (null != fileItemMap) {
            parameterNames.addAll(fileItemMap.keySet());
        }

        return new IteratorEnumeration(parameterNames.iterator());
    }

    @Override
    public String[] getParameterValues(String key) {
        if (null != fileItemMap) {
            List<String> parameterValues = new ArrayList<String>();
            @SuppressWarnings("unchecked")
            Collection<FileItem> fileItems = (Collection<FileItem>) fileItemMap.get(key);
            if (null == fileItems) {
                return null;
            }
            for (FileItem fileItem: fileItems) {
                String contentType = fileItem.getContentType();
                parameterValues.add(getStringFromBytesWithContentType(fileItem.get(), contentType, getRequest().getCharacterEncoding()));
            }
            return parameterValues.toArray(new String[parameterValues.size()]);
        }
        return super.getParameterValues(key);
    }

    static String getStringFromBytesWithContentType(byte[] bytes, String contentType, String requestEncoding) {
        try {
            String charset = getCharsetFromContentType(contentType);
            if (null == charset) {
                charset = requestEncoding;
            }
            if (null == charset) {
                charset = DEFAULT_CHARSET;
            }
            return new String(bytes, charset);
        } catch (UnsupportedEncodingException ignored) {
        }
        try {
            return new String(bytes, DEFAULT_CHARSET);
        } catch (UnsupportedEncodingException never) {
            throw new UnhandledException(never);
        }
    }

    static String getCharsetFromContentType(String contentType) {
        String charset = null;
        if (null != contentType) {
            Perl5Util perl5Util = new Perl5Util();
            if (perl5Util.match("/\\bcharset=\"?(\\S+?)\"?(?:$|;|\\s)/", contentType)) {
                charset = perl5Util.group(1);
            }
        }
        return charset;
    }

    public interface DataSourceFileItem extends FileItem, DataSource {

    }

    private class BaseNameFileItem extends FileItemWrapper {

        private BaseNameFileItem(FileItem fileItem) {
            super(fileItem);
        }

        @Override
        public String getName() {
            String filename = fileItem.getName();
            if (null != filename) {
                filename = filename.substring(filename.lastIndexOf('/') + 1);
                filename = filename.substring(filename.lastIndexOf('\\') + 1);
            }
            return filename;
        }
    }

    private class FileItemWrapper implements DataSourceFileItem {

        FileItem fileItem;

        private FileItemWrapper(FileItem fileItem) {
            this.fileItem = fileItem;
        }

        public String getContentType() {
            return fileItem.getContentType();
        }

        public String getFieldName() {
            return fileItem.getFieldName();
        }

        public void setFieldName(String s) {
            fileItem.setFieldName(s);
        }

        public InputStream getInputStream() throws IOException {
            return fileItem.getInputStream();
        }

        public String getName() {
            return fileItem.getName();
        }

        public OutputStream getOutputStream() throws IOException {
            return fileItem.getOutputStream();
        }

        public long getSize() {
            return fileItem.getSize();
        }

        public String getString() {
            return fileItem.getString();
        }

        public boolean isFormField() {
            return fileItem.isFormField();
        }

        public void setFormField(boolean b) {
            fileItem.setFormField(b);
        }

        public boolean isInMemory() {
            return fileItem.isInMemory();
        }

        public byte[] get() {
            return fileItem.get();
        }

        public void delete() {
            fileItem.delete();
        }

        public String getString(String s) throws UnsupportedEncodingException {
            return fileItem.getString(s);
        }

        public void write(File file) throws Exception {
            fileItem.write(file);
        }
    }

}