package com.imcode.util;

import org.apache.commons.collections.Predicate;
import org.apache.commons.collections.Transformer;
import org.apache.commons.lang.StringEscapeUtils;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

public class HtmlBuilder {

    public CharSequence select(Object name, final Collection values, final Transformer toStringPair,
                               final Predicate selected) {
        return select(name, values, toStringPair, selected, 1) ;
    }

    public CharSequence select(Object name, final Collection values, final Transformer toStringPair,
                               final Predicate selected, int size) {
        return element("select", new String[][] { { "name", name.toString() }, { "size", ""+size } }, options(values, toStringPair, selected));
    }

    private CharSequence emptyElement(String name, String[][] attributes) {
        return element(name, attributes, null) ;
    }

    public CharSequence element(String name, String[][] attributes, CharSequence content) {
        StringBuffer buffer = new StringBuffer(unclosedStartTag(name, attributes));
        if (null != content && content.length() > 0 ) {
            buffer.append('>') ;
            buffer.append(content) ;
            buffer.append(endTag(name)) ;
            return buffer ;
        }
        buffer.append(" />") ;
        return buffer;
    }

    private CharSequence unclosedStartTag(String name, String[][] attributes) {
        StringBuffer buffer = new StringBuffer("<" + name);
        buffer.append(attributes(attributes));
        return buffer;
    }

    private CharSequence attributes(String[][] attributes) {
        StringBuffer attributesBuffer = new StringBuffer();
        if ( null != attributes ) {
            for ( int i = 0; i < attributes.length; i++ ) {
                String[] attribute = attributes[i];
                if ( null != attribute && attribute.length > 0 && null != attribute[0] ) {
                    attributesBuffer.append(' ').append(attribute[0]);
                    if ( attribute.length > 1 && null != attribute[1] ) {
                        attributesBuffer.append("=\"").append(escape(attribute[1])).append('"');
                    }
                }
            }
        }
        return attributesBuffer;
    }

    public CharSequence options(Collection values, Transformer toStringPair) {
        return options(values, toStringPair, null) ;
    }
    
    public CharSequence options(Collection values, Transformer toStringPair, Predicate selected) {
        StringBuffer options = new StringBuffer();
        for ( Object object : values ) {
            String[] stringPair = (String[]) toStringPair.transform(object);
            List<String[]> attributes = new ArrayList<String[]>();
            attributes.add(new String[] { "value", stringPair[0] });
            if ( null != selected && selected.evaluate(object) ) {
                attributes.add(new String[] { "selected" });
            }
            options.append(element("option", attributes.toArray(new String[attributes.size()][]), escape(stringPair[1])));
        }
        return options;
    }

    private CharSequence endTag(String tagName) {
        return "</" + tagName + ">";
    }

    public CharSequence text(Object name, String value) {
        return input("text", name, value) ;
    }

    public CharSequence checkbox(Object name, boolean checked) {
        return input("checkbox", name, checked ? "checked" : null) ;
    }

    public CharSequence submit(Object name, String text) {
        return input("submit", name, text) ;
    }

    public CharSequence escape(CharSequence text) {
        return null == text ? "" : StringEscapeUtils.escapeHtml(text.toString());
    }

    public CharSequence hidden(Object name, Object value) {
        return input("hidden", name, value);
    }

    protected CharSequence input(String type, Object name, Object value) {
        return emptyElement("input", new String[][] {
                { "type", type },
                { "name", name.toString() },
                { "value", null == value ? null : value.toString() },
        }) ;
    }

    public CharSequence textarea(Object name, Object value, int rows, int columns) {
        if (null == value) {
            value = "" ;
        }
        return element("textarea",
                       new String[][] {
                            { "name", name.toString() },
                            { "rows", ""+rows },
                            { "cols", ""+columns }
                        }, "\n"+value.toString() ) ;
    }

    public CharSequence password(Object name) {
        return input("password", name, "") ;
    }
}
