package com.imcode.net.ldap;

import javax.naming.*;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.InitialLdapContext;
import javax.naming.ldap.LdapContext;
import javax.naming.ldap.LdapName;
import java.util.Hashtable;
import java.util.concurrent.Semaphore;

class LdapContextOperationWrapper {

    private static final Semaphore CONTEXTS_LIMITER = new Semaphore(20, true);

    private LdapContext ctx;
    private final Hashtable<String, String> env;

    private LdapContextOperationWrapper(Hashtable<String, String> env) throws LdapClientException {
        this.env = env;
        createLdapContext();
    }

    static LdapContextOperationWrapper from(Hashtable<String, String> env) throws LdapClientException {
        return new LdapContextOperationWrapper(env);
    }

    static boolean checkConnection(Hashtable<String, String> env) throws LdapClientException {
        try {
            new LdapContextOperationWrapper(env).ctx.close();
        } catch (AuthenticationException ae) {
            return false;
        } catch (NamingException e) {
            throw new LdapClientException("", e);
        }
        return true;
    }

    NamingEnumeration<SearchResult> searchAndClose(final String searchFilterExpr, final Object[] parameters, final SearchControls searchControls) throws NamingException {
        try {
            return ctx.search("", searchFilterExpr, parameters, searchControls);
        } finally {
            closeContext();
        }
    }

    NamingEnumeration<SearchResult> searchAndClose(final String searchFilterExpr, final SearchControls searchControls) throws NamingException {
        try {
            return ctx.search("", searchFilterExpr, searchControls);
        } finally {
            closeContext();
        }
    }

    NamingEnumeration searchBasedOnNamespacesAndClose(final String userDn, SearchControls userSearchCtls) throws NamingException {
        try {
            Name userDnName = new LdapName(userDn);
            Name ctxName = new LdapName(ctx.getNameInNamespace());
            Name name = userDnName.getSuffix(ctxName.size());
            return ctx.search(name, "(objectClass=user)", userSearchCtls);
        } finally {
            closeContext();
        }
    }

    private void createLdapContext() throws LdapClientException {
        try {
            CONTEXTS_LIMITER.acquire();
            ctx = new InitialLdapContext(env, null);
        } catch (AuthenticationException ex) {
            throw new LdapAuthenticationException("Authentication failed, using login: '" + env.get(Context.SECURITY_PRINCIPAL) + "'", ex);
        } catch (NameNotFoundException ex) {
            throw new LdapClientException("Root not found: " + env.get(Context.PROVIDER_URL), ex);
        } catch (NamingException ex) {
            throw wrapNamingException(env.get(Context.PROVIDER_URL), ex);
        } catch (InterruptedException e) {
            throw new LdapClientException("Problems with connection freezing ", e);
        }
    }

    private void closeContext() throws NamingException {
        ctx.close();
        CONTEXTS_LIMITER.release();
    }

    private LdapClientException wrapNamingException(String ldapUrl, NamingException ex) {
        return new LdapClientException("Failed to create LDAP context " + ldapUrl + ": " + ex.getExplanation(), ex);
    }

}
