package com.imcode.net.ldap;

import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.UnhandledException;
import org.apache.commons.collections.map.CaseInsensitiveMap;
import org.apache.log4j.Logger;

import javax.naming.directory.DirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.directory.Attribute;
import javax.naming.directory.InitialDirContext;
import javax.naming.*;
import java.util.*;

public class LdapConnection {

    private static final Logger LOG = Logger.getLogger(LdapConnection.class);

    private static final String AUTHENTICATION_TYPE_SIMPLE = "simple";
    private static final String DISTINGUISHED_NAME = "dn";

    private final String ldapUrl;
    private final String ldapBindDn;
    private final String ldapPassword;

    private DirContext ctx;

    public LdapConnection(String ldapUrl, String ldapBindDn, String ldapPassword) throws LdapClientException {
        this.ldapUrl = ldapUrl;
        this.ldapBindDn = ldapBindDn;
        this.ldapPassword = ldapPassword;
        connect();
    }

    private void connect() throws LdapClientException {
        InitialDirContext result;
        try {
            result = new InitialDirContext(createLdapJndiEnvironment(ldapUrl, ldapBindDn, ldapPassword));
        } catch ( AuthenticationException ex ) {
            throw new LdapAuthenticationException( "Authentication failed, using login: '" + ldapBindDn + "'", ex );
        } catch ( NameNotFoundException ex ) {
            throw new LdapClientException( "Root not found: " + ldapUrl, ex );
        } catch ( NamingException ex ) {
            throw wrapNamingException(ldapUrl, ex);
        }
        ctx = result;
    }

    private static Hashtable createLdapJndiEnvironment(String ldapUrl, String ldapBindDn,
                                                       String ldapPassword
    ) {
        Hashtable env = new Hashtable();
        env.put( Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory" );
        env.put( Context.PROVIDER_URL, ldapUrl );
        env.put( Context.SECURITY_AUTHENTICATION, AUTHENTICATION_TYPE_SIMPLE );
        env.put( Context.SECURITY_PRINCIPAL, ldapBindDn );
        env.put( Context.SECURITY_CREDENTIALS, ldapPassword );
        return env;
    }


    private Iterator<Map<String,String>> trySearch(String searchFilterExpr, Object[] parameters, SearchControls searchControls) throws NamingException {
        if (null == searchControls) {
            searchControls = new SearchControls();
        }
        searchControls.setReturningObjFlag(true);
        final NamingEnumeration<SearchResult> enumeration = ctx.search( "", searchFilterExpr, parameters, searchControls );
        return new SearchResultIterator(enumeration, searchControls);
    }

    public Iterator<Map<String,String>> search(String searchFilterExpr, Object[] parameters,
                                     SearchControls searchControls) throws LdapClientException {
        try {
            try {
                return trySearch(searchFilterExpr, parameters, searchControls);
            } catch( CommunicationException ce) {
                LOG.warn( "Problem communicating with LDAP server, retrying.", ce );
                connect();
                return trySearch(searchFilterExpr, parameters, searchControls);
            }
        } catch (NamingException ne) {
            throw new LdapClientException("LDAP search failed.",ne);
        }
    }

    public void close() {
        try {
            ctx.close();
        } catch ( NamingException ne ) {
            LOG.debug("Closing context failed.", ne);
        }
    }

    protected void finalize() throws Throwable {
        super.finalize();
        close();
    }

    private static LdapClientException wrapNamingException(String ldapUrl, NamingException ex) {
        return new LdapClientException( "Failed to create LDAP context " + ldapUrl + ": " + ex.getExplanation(), ex );
    }

    private static class SearchResultIterator implements Iterator<Map<String,String>> {

        private final NamingEnumeration<SearchResult> enumeration;
        private final SearchControls searchControls;

        SearchResultIterator(NamingEnumeration<SearchResult> enumeration, SearchControls searchControls) {
            this.enumeration = enumeration;
            this.searchControls = searchControls;
        }

        public boolean hasNext() {
            try {
                return enumeration.hasMore();
            } catch ( NamingException e ) {
                throw new UnhandledException(e);
            }
        }

        public Map<String, String> next() {
            try {
                return createMapFromSearchResult(enumeration.next());
            } catch ( NamingException e ) {
                throw new UnhandledException(e);
            }
        }

        public void remove() {
            throw new UnsupportedOperationException();
        }

        private Map<String,String> createMapFromSearchResult( SearchResult searchResult ) throws NamingException {

            NamingEnumeration attribEnum = searchResult.getAttributes().getAll();
            Map<String, String> attributes = new CaseInsensitiveMap();
            while ( attribEnum.hasMoreElements() ) {
                Attribute attribute = (Attribute)attribEnum.nextElement();
                String attributeName = attribute.getID();
                String attributeValue = attribute.get().toString();
                attributes.put( attributeName, attributeValue );
            }
            if (!attributes.containsKey(DISTINGUISHED_NAME)) {
                boolean includeDistinguishedName = null != searchControls && searchControls.getReturningObjFlag() && ( null == searchControls.getReturningAttributes()
                                                   || ArrayUtils.contains(searchControls.getReturningAttributes(), DISTINGUISHED_NAME ));
                if ( includeDistinguishedName  ) {
                    DirContext dirContext = (DirContext)searchResult.getObject();
                    String distinguishedName = dirContext.getNameInNamespace();
                    attributes.put( DISTINGUISHED_NAME, distinguishedName );
                }
            }

            return attributes;
        }

    }
}
