/*
 * Decompiled with CFR 0.152.
 */
package imcode.server.document.index;

import imcode.server.document.DocumentDomainObject;
import imcode.server.document.index.BackgroundIndexBuilder;
import imcode.server.document.index.DefaultDirectoryIndex;
import imcode.server.document.index.DirectoryIndex;
import imcode.server.document.index.DocumentIndex;
import imcode.server.document.index.DocumentQuery;
import imcode.server.document.index.IndexDocumentFactory;
import imcode.server.document.index.IndexException;
import imcode.server.document.index.IndexRebuildTimerTask;
import imcode.server.user.UserDomainObject;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.filefilter.FileFilterUtils;
import org.apache.log4j.Logger;
import org.apache.lucene.index.IndexReader;

public class RebuildingDirectoryIndex
implements DocumentIndex {
    private static final Logger log = Logger.getLogger((String)RebuildingDirectoryIndex.class.getName());
    private final BackgroundIndexBuilder backgroundIndexBuilder;
    private final long indexRebuildSchedulePeriodInMilliseconds;
    private final Timer scheduledIndexRebuildTimer = new Timer(true);
    private IndexRebuildTimerTask currentIndexRebuildTimerTask;
    private DirectoryIndex index = new NullDirectoryIndex();

    public RebuildingDirectoryIndex(File indexParentDirectory, float indexRebuildSchedulePeriodInMinutes, IndexDocumentFactory indexDocumentFactory) {
        this.indexRebuildSchedulePeriodInMilliseconds = (long)(indexRebuildSchedulePeriodInMinutes * 60000.0f);
        this.backgroundIndexBuilder = new BackgroundIndexBuilder(indexParentDirectory, this, indexDocumentFactory);
        File indexDirectory = RebuildingDirectoryIndex.findLatestIndexDirectory(indexParentDirectory);
        long indexModifiedTime = 0L;
        if (null != indexDirectory) {
            indexModifiedTime = indexDirectory.lastModified();
            this.index = new DefaultDirectoryIndex(indexDirectory, indexDocumentFactory);
        } else {
            this.rebuildBecauseOfError("No existing index.", null);
        }
        if (this.isSchedulingIndexRebuilds()) {
            log.info((Object)("First index rebuild scheduled at " + RebuildingDirectoryIndex.formatDatetime(this.restartIndexRebuildScheduling(indexModifiedTime))));
        } else {
            log.info((Object)"Scheduling of index rebuilds is disabled.");
        }
    }

    private synchronized Date restartIndexRebuildScheduling(long indexModifiedTime) {
        if (!this.isSchedulingIndexRebuilds()) {
            return null;
        }
        long time = System.currentTimeMillis();
        Date nextTime = new Date(indexModifiedTime + this.indexRebuildSchedulePeriodInMilliseconds);
        if (nextTime.getTime() < time) {
            nextTime.setTime(time);
        }
        this.restartIndexRebuildScheduling(nextTime);
        return nextTime;
    }

    private synchronized void restartIndexRebuildScheduling(Date nextTime) {
        if (null != this.currentIndexRebuildTimerTask) {
            this.currentIndexRebuildTimerTask.cancel();
            log.trace((Object)"Canceled existing index rebuild timer task.");
        }
        try {
            log.debug((Object)("Restarting scheduling of index rebuilds. First rebuild at " + RebuildingDirectoryIndex.formatDatetime(nextTime) + "."));
            this.backgroundIndexBuilder.touchIndexParentDirectory();
            this.currentIndexRebuildTimerTask = new IndexRebuildTimerTask(this.indexRebuildSchedulePeriodInMilliseconds, this.backgroundIndexBuilder);
            this.scheduledIndexRebuildTimer.scheduleAtFixedRate((TimerTask)this.currentIndexRebuildTimerTask, nextTime, this.indexRebuildSchedulePeriodInMilliseconds);
        }
        catch (IllegalStateException ise) {
            log.error((Object)"Failed to start index rebuild scheduling.", (Throwable)ise);
        }
    }

    private boolean isSchedulingIndexRebuilds() {
        return this.indexRebuildSchedulePeriodInMilliseconds > 0L;
    }

    static File findLatestIndexDirectory(File indexParentDirectory) {
        try {
            if (indexParentDirectory.exists() && !indexParentDirectory.isDirectory()) {
                log.debug((Object)("Deleting non-directory " + indexParentDirectory));
                FileUtils.forceDelete((File)indexParentDirectory);
            }
            if (!indexParentDirectory.exists()) {
                log.debug((Object)("Creating directory " + indexParentDirectory));
                FileUtils.forceMkdir((File)indexParentDirectory);
            }
            File[] indexDirectories = indexParentDirectory.listFiles((FileFilter)FileFilterUtils.directoryFileFilter());
            RebuildingDirectoryIndex.sortFilesByLastModifiedWithLatestFirst(indexDirectories);
            File indexDirectory = null;
            for (int i = 0; i < indexDirectories.length; ++i) {
                File directory = indexDirectories[i];
                if (IndexReader.indexExists((File)directory)) {
                    if (null == indexDirectory) {
                        log.debug((Object)("Found index in directory " + directory));
                        indexDirectory = directory;
                        continue;
                    }
                    log.debug((Object)("Deleting old index directory " + directory));
                    FileUtils.forceDelete((File)directory);
                    continue;
                }
                log.debug((Object)("Deleting non-index directory " + directory));
                FileUtils.forceDelete((File)directory);
            }
            return indexDirectory;
        }
        catch (IOException ioe) {
            throw new IndexException(ioe);
        }
    }

    private static void sortFilesByLastModifiedWithLatestFirst(File[] indexDirectories) {
        Arrays.sort(indexDirectories, new Comparator(){

            public int compare(Object o1, Object o2) {
                File f1 = (File)o1;
                File f2 = (File)o2;
                return new Long(f2.lastModified()).compareTo(new Long(f1.lastModified()));
            }
        });
    }

    static String formatDatetime(Date nextExecutionTime) {
        return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(nextExecutionTime);
    }

    public void indexDocument(DocumentDomainObject document) {
        log.debug((Object)"Adding document.");
        this.backgroundIndexBuilder.addDocument(document);
        try {
            this.index.indexDocument(document);
        }
        catch (IndexException e) {
            this.rebuildBecauseOfError("Failed to add document " + document.getId() + " to index.", e);
        }
    }

    public void removeDocument(DocumentDomainObject document) {
        log.debug((Object)"Removing document.");
        this.backgroundIndexBuilder.removeDocument(document);
        try {
            this.index.removeDocument(document);
        }
        catch (IndexException e) {
            this.rebuildBecauseOfError("Failed to remove document " + document.getId() + " from index.", e);
        }
    }

    public List search(DocumentQuery query, UserDomainObject searchingUser) throws IndexException {
        try {
            List documents = this.index.search(query, searchingUser);
            if (this.index.isInconsistent()) {
                this.rebuildBecauseOfError("Index is inconsistent.", null);
            }
            return documents;
        }
        catch (IndexException ex) {
            this.rebuildBecauseOfError("Search failed.", ex);
            return Collections.EMPTY_LIST;
        }
    }

    private void rebuildBecauseOfError(String message, IndexException ex) {
        log.error((Object)(message + " Starting index rebuild."), (Throwable)ex);
        this.rebuild();
    }

    public void rebuild() {
        if (this.isSchedulingIndexRebuilds()) {
            this.restartIndexRebuildScheduling(new Date());
        } else {
            this.backgroundIndexBuilder.start();
        }
    }

    void notifyRebuildComplete(DirectoryIndex newIndex) {
        DirectoryIndex oldIndex = this.index;
        this.index = newIndex;
        if (!oldIndex.equals(this.index)) {
            oldIndex.delete();
        }
    }

    private static class NullDirectoryIndex
    implements DirectoryIndex {
        private NullDirectoryIndex() {
        }

        public boolean isInconsistent() {
            return false;
        }

        public void delete() {
        }

        public void indexDocument(DocumentDomainObject document) throws IndexException {
        }

        public void removeDocument(DocumentDomainObject document) throws IndexException {
        }

        public List search(DocumentQuery query, UserDomainObject searchingUser) throws IndexException {
            return Collections.EMPTY_LIST;
        }

        public void rebuild() throws IndexException {
        }
    }
}

