package com.imcode.imcms.addon.imsurvey.oneflow;

import com.google.gson.*;
import com.imcode.imcms.addon.imsurvey.SystemProperties;
import com.imcode.imcms.api.DatabaseService;
import org.apache.commons.dbutils.DbUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;

import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.imcode.imcms.addon.imsurvey.SystemProperties.*;
import static com.imcode.imcms.addon.imsurvey.utils.OneflowFields.*;

/**
 * Oneflow API service
 * Created by dmizem from Ubrainians for ImCode
 *
 * @author dmizem.
 */
public class OneflowService {
    private static final Logger log = Logger.getLogger(OneflowService.class);

    private static final int FAILURE_HTTP_RESPONSE_CODE = 390;

	private static final OneflowService instance = new OneflowService();

    private final Map<String, Document> templateCache = new HashMap<>();
    private final Map<String, Document> templateGroupFieldCache = new HashMap<>();
    private final Map<String, Document> templateGroupCache = new HashMap<>();
    private final Map<String, List<Field>> templateFieldCache = new HashMap<>();

    private boolean isSwappedToEmail = false;

    private OneflowService() {
    }

    public static OneflowService getInstance() {
        return instance;
    }

    public boolean isSurveyInSyncWithOneflow(int surveyId, DatabaseService databaseService) {
        // TODO: 31.10.18 Removed some commented code, maybe be required in future so watch commit history for hints.
        // TODO Probably remove since, oneflow template groups are not the same as on page and not correspond actual agreement template or try a way to implement it
        return true;
    }

    private String getDocumentURI(String signSuccessUrl, String templateId, String templateTypeId, Map<String, String> props, HttpServletRequest request) throws Exception {
        if (props.containsKey(PROPERTY_IM_SURVEY_ID)) {
            try {
                props.remove(PROPERTY_IM_SURVEY_ID);
            } catch (NumberFormatException e) {
                e.printStackTrace();
            }
        }

        if (templateId == null || templateId.length() <= 4) {
            throw new IllegalArgumentException(ERROR_WRONG_TEMPLATE_ID + templateId);
        }

	    HttpURLConnection connection = getHttpURLConnection(ONEFLOW_BASE_URL + API_TEMPLATES_URL + templateId, "GET");

        int responseCode = connection.getResponseCode();

        boolean parserError = false;
        long docId = 0;
        JsonObject fetchedAgreement = null;

        if (responseCode > FAILURE_HTTP_RESPONSE_CODE) {
            String error = IOUtils.toString(connection.getErrorStream());
            connection.disconnect();
            log.error(ONEFLOW_BASE_URL + API_CONTRACTS_URL + templateId + ":"
		            + connection.getResponseCode() + connection.getResponseMessage() + "\n" + error);
            throw new IOException(error);
        } else {
            JsonElement parsedReturn = new JsonParser().parse(new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8));
            connection.disconnect();
            if (parsedReturn != null && parsedReturn.isJsonObject()) {
                fetchedAgreement = parsedReturn.getAsJsonObject();

	            JsonObject templateType = fetchedAgreement.getAsJsonObject(JSON_TEMPLATE_TYPE);
	            docId = templateType.isJsonNull() ? 0 : templateType.get(JSON_ID).getAsLong();
            } else {
                parserError = true;
            }
        }

        if (parserError) {
            throw new Exception(ERROR_TEMPLATE_CHANGED);
        } else {
            // Update the document with form data
            if (docId > 0) {
                String urlParameters = fetchedAgreement.toString();

                //Retrieving agreement data
	            JsonObject contract = buildContract(templateId, templateTypeId, props);

	            connection = getHttpURLConnection(ONEFLOW_BASE_URL + API_CONTRACTS_URL + API_CONTRACTS_CREATE_ENDPOINT);
	            connection.addRequestProperty("x-oneflow-user-email", ONEFLOW_SECURITY_TOKEN_EMAIL);
				connection.setRequestProperty("Content-Type", "application/json");
                connection.setRequestProperty("Content-Length", "" + urlParameters.getBytes().length);

                OutputStreamWriter streamWriter = new OutputStreamWriter(connection.getOutputStream());
                streamWriter.write(contract.toString());
                streamWriter.close();

                responseCode = connection.getResponseCode();
                log.debug(ONEFLOW_BASE_URL + API_CONTRACTS_URL + ":" + responseCode + connection.getResponseMessage());

                if (responseCode > FAILURE_HTTP_RESPONSE_CODE) {
                    log.error(ONEFLOW_BASE_URL + API_CONTRACTS_URL + ":" + responseCode + connection.getResponseMessage() + "\n" + IOUtils.toString(connection.getErrorStream()));
                    throw new IOException(IOUtils.toString(connection.getErrorStream()));
                }

                JsonElement agreementResponse = new JsonParser().parse(new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8));
                connection.disconnect();

                //Getting id and publishing agreement
                if (null != agreementResponse && agreementResponse.isJsonObject()) {
                    JsonPrimitive id = agreementResponse.getAsJsonObject().getAsJsonPrimitive(JSON_ID);
                    int agreementId = id.getAsInt();

                    if (agreementId > 0) {
                        connection = getHttpURLConnection(ONEFLOW_BASE_URL + API_CONTRACTS_URL + agreementId + API_CONTRACTS_PUBLISH_ENDPOINT);
                        connection.setRequestProperty("Content-Type", "application/json");
                        connection.setRequestProperty("Content-Length", "" + urlParameters.getBytes().length);
						connection.setRequestProperty("x-oneflow-user-email", ONEFLOW_SECURITY_TOKEN_EMAIL);

                        JsonObject publishContent = new JsonObject();
                        publishContent.addProperty(JSON_SUBJECT, PUBLICATION_SUBJECT);
                        publishContent.addProperty(JSON_MESSAGE, PUBLICATION_MESSAGE);

                        OutputStreamWriter outputStreamWriter = new OutputStreamWriter(connection.getOutputStream());
                        outputStreamWriter.write(publishContent.toString());
                        outputStreamWriter.close();
                        connection.disconnect();

                        if (connection.getResponseCode() < FAILURE_HTTP_RESPONSE_CODE) {
                            log.info("Agreement with id " + agreementId + " was published: " + connection.getResponseCode() + ":" + connection.getResponseMessage());

                            DeliveryChannel deliveryChannel = getDeliveryChannelFromContract(contract, 1);
                            if (null == deliveryChannel) {
                                log.error(ERROR_OBTAIN_DELIVERY_METHOD + agreementId);
                                connection.disconnect();
                                throw new IOException(ERROR_OBTAIN_DELIVERY_METHOD + agreementId);
                            }
                            if (deliveryChannel == DeliveryChannel.NONE || isSwappedToEmail) {
                                // Generating user token and creating appropriate URl to redirect
                                JsonElement publishResponse = new JsonParser().parse(new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8));
                                if (publishResponse.isJsonObject()) {
                                    String participantId = getParticipantId(publishResponse.getAsJsonObject());
                                    String accessLink = getAccessLink(Integer.toString(agreementId), participantId);
                                    if (!accessLink.isEmpty()) {
                                        log.debug("Agreement id:" + agreementId + "\t Participant id: " + participantId + "\t Access link:" + accessLink);
                                        return accessLink;
                                    }
                                }
                            } else {
                                return signSuccessUrl;
                            }
                        } else {
                            log.error("Agreement with id " + agreementId + " wasn't published: " + connection.getResponseCode() + ":" + connection.getResponseMessage() + "\n" + IOUtils.toString(connection.getErrorStream()));
                            connection.disconnect();
                            throw new IOException(IOUtils.toString(connection.getErrorStream()));
                        }
                    }
                }
                connection.disconnect();
            }
        }
        return null;
    }

	private int getWorkspaceId(Integer templateId, Integer templateTypeId) throws IOException {
		int workspaceId = 0;
		HttpURLConnection dataToCreateContractRequest = getHttpURLConnection(ONEFLOW_BASE_URL +
				API_HELPERS_URL + API_CREATE_CONTRACT_HELPER_ENDPOINT + templateTypeId, "GET");
		dataToCreateContractRequest.addRequestProperty("x-oneflow-user-email", ONEFLOW_SECURITY_TOKEN_EMAIL);
		if (dataToCreateContractRequest.getResponseCode() == 200) {
			JsonElement data = new JsonParser().parse(new InputStreamReader(dataToCreateContractRequest.getInputStream(), StandardCharsets.UTF_8));
			dataToCreateContractRequest.disconnect();

			for (JsonElement workspace : data.getAsJsonArray()) {
				JsonArray templates = workspace.getAsJsonObject().get(JSON_TEMPLATES).getAsJsonArray();
				for (JsonElement template : templates) {
					Integer id = template.getAsJsonObject().get(JSON_ID).getAsInt();
					if (id.equals(templateId)) {
						workspaceId = workspace.getAsJsonObject().get(JSON_ID).getAsInt();
					}
				}
			}
		} else {
			throw new IOException(IOUtils.toString(dataToCreateContractRequest.getErrorStream()));
		}
		return workspaceId;
	}

	private JsonObject buildContract(String templateId, String templateTypeId, Map<String, String> props) throws IOException {
        JsonObject contract = new JsonObject();
        contract.addProperty(JSON_TEMPLATE_ID, Integer.valueOf(templateId));
		contract.addProperty(JSON_WORKSPACE_ID, getWorkspaceId(Integer.valueOf(templateId), Integer.valueOf(templateTypeId)));

        //  Adding participants data
        JsonArray parties = new JsonArray();

        List<OneflowParticipant> participantsData = new ArrayList<>();

        for (Iterator<Map.Entry<String, String>> it = props.entrySet().iterator(); it.hasNext(); ) {
            Map.Entry<String, String> entry = it.next();
            if (!entry.getValue().isEmpty()) {
                if (entry.getKey().matches(REGEX_USER_FIELD)) {
                    String[] splittedData = entry.getKey().split(REGEX_USER_DIVIDER);
                    String userId = splittedData[1].substring(0, splittedData[1].indexOf('_'));
                    String fieldName = splittedData[1].substring(splittedData[1].indexOf('_') + 1);
                    if (!userId.isEmpty() && !fieldName.isEmpty()) {
                        OneflowParticipant oneflowParticipant = null;

                        boolean isNew = true;
                        for (OneflowParticipant op : participantsData) {
                            if (op.getId().equals(userId)) {
                                oneflowParticipant = op;
                                isNew = false;
                            }
                        }
                        if (oneflowParticipant == null) {
                            oneflowParticipant = new OneflowParticipant(userId);
                        }

                        if (UserField.isMember(fieldName.toLowerCase())) {

                            UserField userField = UserField.valueOf(fieldName.toUpperCase());
                            if (userField.equals(UserField.DELIVERY_CHANNEL)) {
                                oneflowParticipant.addValues(userField, String.valueOf(DeliveryChannel.valueOf(entry.getValue())));
                            }

                            if (userField.equals(UserField.SIGN_METHOD)) {
                                oneflowParticipant.addValues(userField, String.valueOf(SignMethod.valueOf(entry.getValue())));

                            } else {
                                oneflowParticipant.addValues(userField, entry.getValue());
                            }
                        }
                        if (isNew) {
                            participantsData.add(oneflowParticipant);
                        }
                    }
                    it.remove();
                }
            }
        }

        // Users in order as it must be
        Collections.sort(participantsData);

        for (OneflowParticipant entity : participantsData) {
            JsonObject partyClient = new JsonObject();
	        JsonArray participants = new JsonArray();
			JsonObject participant = new JsonObject();
            for (Map.Entry<Enum, String> entry : entity.getValues().entrySet()) {
	            Enum key = entry.getKey();
	            String value = entry.getValue();

	            //oneflow has two same properties but different context :)
	            if (key.equals(UserField.COMPANY_NAME)) {
		            partyClient.addProperty(UserField.NAME.getValue(), value);
	            } else if (key.equals(UserField.IDENTIFICATION_NUMBER) || key.equals(UserField.COUNTRY_CODE)) {
		            partyClient.addProperty(((UserField) key).getValue(), value);
	            } else if (key.equals(UserField.DELIVERY_CHANNEL) && DeliveryChannel.isMember(entry.getValue())) {
		            if (DeliveryChannel.valueOf(value) == DeliveryChannel.NONE) {
			            isSwappedToEmail = true;
		            }
		            participant.addProperty(((UserField) key).getValue(), entry.getValue().toLowerCase());
	            } else if (key.equals(UserField.TYPE)) {
		            OneflowParticipant.Type type = OneflowParticipant.Type.valueOf(Integer.parseInt(value));
		            participant.addProperty(JSON_SIGNATORY, type.isSignatory());
		            JsonObject permissions = new JsonObject();
		            permissions.addProperty(JSON_CONTRACT_UPDATE, type.isContractUpdate());
		            participant.add(JSON_PERMISSIONS, permissions);
	            } else if (key.equals(UserField.SIGN_METHOD)) {
		            participant.addProperty(((UserField) key).getValue(), value.toLowerCase());
	            } else {
		            participant.addProperty(((UserField) key).getValue(), value);
	            }
            }

            // Removing not user data w/o all required information form request
            JsonElement tmpName = participant.get(UserField.NAME.getValue());
            JsonElement tmpEmail = participant.get(UserField.EMAIL.getValue());
            if ((null != tmpName && !tmpName.getAsString().isEmpty()) && (null != tmpEmail && !tmpEmail.getAsString().isEmpty())) {
				participants.add(participant);
                partyClient.add(JSON_PARTICIPANTS, participants);
	            partyClient.addProperty(JSON_TYPE, JSON_TYPE_COMPANY);
                parties.add(partyClient);
            }

        }
		addContractCreator(contract);
        contract.add(JSON_PARTIES, parties);

        // Adding contract data
        JsonArray dataList = new JsonArray();

        for (Map.Entry<String, String> entry : props.entrySet()) {
            if (!entry.getValue().isEmpty()) {
                JsonObject dataItemValue = new JsonObject();
                dataItemValue.addProperty(JSON_CUSTOM_ID, entry.getKey());
                dataItemValue.addProperty(JSON_VALUE, entry.getValue());

                dataList.add(dataItemValue);
            }
        }

        contract.add(JSON_DATA_FIELDS, dataList);
        log.info("contract:" + contract);
        return contract;
    }

	private void addContractCreator(JsonObject contract){
		final JsonObject myParty = new JsonObject();
		myParty.addProperty(UserField.NAME.getValue(), ONEFLOW_CREATOR_NAME);
		myParty.addProperty(UserField.COUNTRY_CODE.getValue(), ONEFLOW_CREATOR_COUNTRY_CODE);
		myParty.addProperty(UserField.IDENTIFICATION_NUMBER.getValue(), ONEFLOW_CREATOR_IDENTIFICATION_NUMBER);

		final JsonArray participants = new JsonArray();
		final OneflowParticipant.Type participant = OneflowParticipant.Type.ORGANIZER;
		final JsonObject creatorParticipant = new JsonObject();

		creatorParticipant.addProperty(JSON_SIGNATORY, participant.isSignatory());
		creatorParticipant.addProperty(JSON_ORGANIZER, participant.isOrganizer());
		creatorParticipant.addProperty(UserField.EMAIL.getValue(), ONEFLOW_SECURITY_TOKEN_EMAIL);

		final JsonObject permissions = new JsonObject();

		permissions.addProperty(JSON_CONTRACT_UPDATE, participant.isContractUpdate());
		creatorParticipant.add(JSON_PERMISSIONS, permissions);
		participants.add(creatorParticipant);

		myParty.add(JSON_PARTICIPANTS, participants);
		contract.add(JSON_MY_PARTY, myParty);
	}

    private String getAccessLink(String agreementId, String userId) throws IOException {
        JsonObject tokenBody = new JsonObject();
        tokenBody.addProperty(JSON_TYPE, JSON_FORM);

	    HttpURLConnection connection = getHttpURLConnection(ONEFLOW_BASE_URL + API_CONTRACTS_URL + agreementId + API_CONTRACTS_PARTICIPANTS_ENDPOINT + userId + API_CONTRACTS_ACCESS_LINK_ENDPOINT);
        connection.setRequestProperty("Content-Type", "application/json");

        OutputStreamWriter streamWriter = new OutputStreamWriter(connection.getOutputStream());
        streamWriter.write(tokenBody.toString());
        streamWriter.close();

        int responseCode = connection.getResponseCode();
        log.debug(ONEFLOW_BASE_URL + API_CONTRACTS_URL + agreementId + "/participants/" + userId + "/tokens " + ":" + responseCode + connection.getResponseMessage());

        if (responseCode > FAILURE_HTTP_RESPONSE_CODE) {
            log.error(ONEFLOW_BASE_URL + API_CONTRACTS_URL + agreementId + "/participants/" + userId + "/tokens " + ":" + responseCode + connection.getResponseMessage() + "\n" + IOUtils.toString(connection.getErrorStream()));
            throw new IOException(IOUtils.toString(connection.getErrorStream()));
        }
        String accessLink = "";
        JsonElement tokenResponse = new JsonParser().parse(new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8));
        connection.disconnect();

        //Getting id and publishing agreement
        if (tokenResponse.isJsonObject()) {
            accessLink = tokenResponse.getAsJsonObject().get(JSON_ACCESS_LINK).getAsString();
        }

        return accessLink;
    }

    private String getParticipantId(JsonObject publishedAgreementResponse) {
        String participantId = "";
        JsonArray parties = publishedAgreementResponse.getAsJsonArray(JSON_PARTIES);
	    if (null != parties) {
            for (JsonElement tmpParty : parties) {
                JsonObject party = tmpParty.getAsJsonObject();
	            if (!party.get(JSON_MY_PARTY).getAsBoolean()) {
		            JsonObject participant = party.getAsJsonArray(JSON_PARTICIPANTS).get(0).getAsJsonObject();
		            if (participant.get(JSON_SIGNATORY).getAsBoolean()) {
			            JsonElement idElement = participant.getAsJsonObject().get(JSON_ID);
			            participantId = idElement.getAsString();
		            }
	            }
            }
        }
        return participantId;
    }

    private DeliveryChannel getDeliveryChannelFromContract(JsonObject contract, Integer participantNumber) {
	    JsonArray userParties = contract.getAsJsonArray(JSON_PARTIES);
	    JsonObject party = userParties.get(0).getAsJsonObject();
	    if (userParties.size() > 1) {
		    party = userParties.get(participantNumber).getAsJsonObject();
	    }

	    JsonElement deliveryChannel = party.getAsJsonArray(JSON_PARTICIPANTS).get(0).getAsJsonObject().get(UserField.DELIVERY_CHANNEL.getValue());
	    return DeliveryChannel.valueOf(deliveryChannel.getAsString().toUpperCase());
    }

    public List<SignMethod> getAvailableSignMethods(String id) throws IOException {
        List<SignMethod> availableMethods = new ArrayList<>();

        HttpURLConnection connection = getHttpURLConnection(ONEFLOW_BASE_URL + API_CONTRACTS_URL + id, "GET");

        int responseCode = connection.getResponseCode();
        if (responseCode == 404) {
            connection.disconnect();
            return null;

        } else if (responseCode > FAILURE_HTTP_RESPONSE_CODE) {
            String error = IOUtils.toString(connection.getErrorStream());
            connection.disconnect();
            log.error(ONEFLOW_BASE_URL + API_CONTRACTS_URL + ":" + responseCode + connection.getResponseMessage() + "\n" + error);
            throw new IOException(error);
        } else {
            JsonElement signMethodsResponse = new JsonParser().parse(new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8));
            connection.disconnect();
            if (signMethodsResponse.isJsonObject()) {
                JsonObject optionsElem = signMethodsResponse.getAsJsonObject().getAsJsonObject(JSON_AVAILABLE_OPTIONS);
                if (null != optionsElem) {
                    JsonArray signMethods = optionsElem.getAsJsonArray(JSON_SIGN_METHODS);
                    if (null != signMethods) {
	                    for (JsonElement signMethod : signMethods) {
		                    SignMethod method = SignMethod.valueOf(((JsonObject) signMethod).get("name").getAsString().toUpperCase());
		                    availableMethods.add(method);
	                    }
                    }
                }
            }
        }

        return availableMethods;
    }

    public List<DeliveryChannel> getAvailableDeliveryChannels(String id) throws IOException {
        List<DeliveryChannel> availableDeliveries = new ArrayList<>();

        HttpURLConnection connection = getHttpURLConnection(ONEFLOW_BASE_URL + API_CONTRACTS_URL + id, "GET");

        int responseCode = connection.getResponseCode();
        if (responseCode == 404) {
            connection.disconnect();
            return null;

        } else if (responseCode > FAILURE_HTTP_RESPONSE_CODE) {
            String error = IOUtils.toString(connection.getErrorStream());
            connection.disconnect();
            log.error(ONEFLOW_BASE_URL + API_CONTRACTS_URL + id + ":" + responseCode + connection.getResponseMessage() + "\n" + error);
            throw new IOException(error);
        } else {
            JsonElement deliveryChannelResponse = new JsonParser().parse(new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8));
            connection.disconnect();
            if (deliveryChannelResponse.isJsonObject()) {
                JsonObject optionsElem = deliveryChannelResponse.getAsJsonObject().getAsJsonObject(JSON_AVAILABLE_OPTIONS);
                if (null != optionsElem) {
                    JsonArray deliveryChannels = optionsElem.getAsJsonArray(JSON_DELIVERY_CHANNELS);
                    if (null != deliveryChannels) {
	                    for (JsonElement deliveryChannel : deliveryChannels) {
		                    DeliveryChannel channel = DeliveryChannel.valueOf(((JsonObject) deliveryChannel).get(JSON_NAME).getAsString().toUpperCase());
		                    availableDeliveries.add(channel);
	                    }
                    }
                }
            }
        }

        return availableDeliveries;
    }

    public List<Document> getTemplates() throws IOException {
        List<Document> templates = new ArrayList<>();

	    HttpURLConnection connection = getHttpURLConnection(ONEFLOW_BASE_URL + API_TEMPLATE_TYPES_URL, "GET", true);
        if (connection.getResponseCode() > FAILURE_HTTP_RESPONSE_CODE) {
            String error = IOUtils.toString(connection.getErrorStream());
            connection.disconnect();
	        log.error(ONEFLOW_BASE_URL + API_TEMPLATE_TYPES_URL + ":" + connection.getResponseCode() + connection.getResponseMessage() + "\n" + error);
            throw new IOException(error);
        } else {
            JsonElement templateGroupsResponse = new JsonParser().parse(new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8));
            connection.disconnect();
            if (templateGroupsResponse.isJsonObject()) {
	            JsonArray templateListElem = ((JsonObject) templateGroupsResponse).getAsJsonArray(JSON_DATA);
                if (null != templateListElem) {
                    for (JsonElement templateElem : templateListElem) {

                        if (null != templateElem && templateElem.isJsonObject()) {
                            JsonObject templateObj = templateElem.getAsJsonObject();
                            //Getting name of template
                            JsonPrimitive title = templateObj.getAsJsonPrimitive(JSON_NAME);
                            JsonPrimitive templateId = templateObj.getAsJsonPrimitive(JSON_ID);

                            if (null != templateId && null != title) {
	                            Document template = new Document();
	                            template.setId(templateId.getAsString());
	                            template.setTitle(title.getAsString());
	                            template.setFields(getTemplateFields(templateId.getAsString()));

	                            templates.add(template);
                            }
                        }
                    }
                }
            }
        }
        return templates;
    }

    public Document getDocument(String id) throws IOException {
        Document document = null;

        HttpURLConnection connection = getHttpURLConnection(ONEFLOW_BASE_URL + API_CONTRACTS_URL + id, "GET");

        int responseCode = connection.getResponseCode();
        if (responseCode == 404) {
            connection.disconnect();
            return null;

        } else if (responseCode > FAILURE_HTTP_RESPONSE_CODE) {
            String error = IOUtils.toString(connection.getErrorStream());
            connection.disconnect();
            log.error(ONEFLOW_BASE_URL + API_CONTRACTS_URL + id + ":" + connection.getResponseCode() + connection.getResponseMessage() + "\n" + error);
            throw new IOException(error);

        } else {
            JsonElement parsedReturn = new JsonParser().parse(new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8));
            connection.disconnect();
            if (parsedReturn.isJsonObject()) {
                document = parseDocument(parsedReturn);
                document.setFields(getTemplateFields(id));
            }
        }

        return document;
    }

    public Document getTemplateGroup(String id) throws IOException {
	    Document templateGroup;
        if (templateGroupCache.get(id) != null) {
            templateGroup = templateGroupCache.get(id);
        } else {
            templateGroup = new Document();
            HttpURLConnection connection = getHttpURLConnection(ONEFLOW_BASE_URL + API_TEMPLATE_TYPES_URL + id, "GET", true);
            if (connection.getResponseCode() > FAILURE_HTTP_RESPONSE_CODE) {
                String error = IOUtils.toString(connection.getErrorStream());
                connection.disconnect();
                log.error(ONEFLOW_BASE_URL + API_TEMPLATE_TYPES_URL + id + ":" + connection.getResponseCode() + connection.getResponseMessage() + "\n" + error);
                throw new IOException(error);
            } else {
	            List<Field> fields = new LinkedList<>();
	            JsonElement templateGroupResponse = new JsonParser().parse(new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8));
	            connection.disconnect();
	            if (templateGroupResponse.isJsonObject()) {
		            JsonObject templateGroupObj = templateGroupResponse.getAsJsonObject();
		            JsonPrimitive title = templateGroupObj.getAsJsonPrimitive(JSON_NAME);
		            JsonPrimitive templateGroupId = templateGroupObj.getAsJsonPrimitive(JSON_ID);
		            if (null != templateGroupId && null != title) {
			            templateGroup.setId(templateGroupId.getAsString());
			            templateGroup.setTitle(title.getAsString());
			            JsonArray templateListElem = templateGroupObj.getAsJsonArray(JSON_DATA_FIELDS);
			            fields.addAll(jsonsTemplateArrayToFields(templateListElem));
		            }
	            }
	            templateGroup.setFields(fields);
	            templateFieldCache.put(id, fields);
	            templateGroupCache.put(id, templateGroup);
            }
        }
        return templateGroup;
    }

	private List<Field> jsonsTemplateArrayToFields(JsonArray templateListElem) {
		final List<Field> fields = new ArrayList<>();
		for (JsonElement templateElem : templateListElem) {
			JsonObject templateObj = templateElem.getAsJsonObject();
			JsonPrimitive name = templateObj.getAsJsonPrimitive(JSON_NAME);
			JsonPrimitive customId = templateObj.getAsJsonPrimitive(JSON_CUSTOM_ID);
			if (null != name && null != customId) {
				fields.add(new Field(Field.Type.TEXT, name.getAsString(), customId.getAsString()));
			}
		}
		return fields;
	}

    /**
     * Method to get list of templates as fields
     *
     * @param id ID of template group
     * @return Document with fields as templates id
     * @throws IOException Error during API reading
     */
    public Document getTemplatesInGroupAsFields(String id) throws IOException {
        Document templateGroup;
        if (templateGroupFieldCache.get(id) != null) {
            templateGroup = templateGroupFieldCache.get(id);
        } else {
            //Getting initial values
            templateGroup = getTemplateGroup(id);
            HttpURLConnection connection = getHttpURLConnection(ONEFLOW_BASE_URL + API_TEMPLATES_URL, "GET", true);
	        connection.addRequestProperty("x-oneflow-user-email", ONEFLOW_SECURITY_TOKEN_EMAIL);
			if (connection.getResponseCode() > FAILURE_HTTP_RESPONSE_CODE) {
                String error = IOUtils.toString(connection.getErrorStream());
                connection.disconnect();
                log.error(ONEFLOW_BASE_URL + API_TEMPLATE_TYPES_URL + id + ":" + connection.getResponseCode() + connection.getResponseMessage() + "\n" + error);
                throw new IOException(error);
            } else {

                JsonElement templateGroupResponse = new JsonParser().parse(new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8));
                connection.disconnect();
                if (null != templateGroupResponse && templateGroupResponse.isJsonObject()) {
                    JsonArray templateGroupCollection = templateGroupResponse.getAsJsonObject().getAsJsonArray(JSON_DATA);
                    if (null != templateGroupCollection) {
                        List<Field> templateList = new ArrayList<>();
	                    for (JsonElement templateElem : templateGroupCollection) {
		                    if (null != templateElem && templateElem.isJsonObject()) {
			                    JsonElement templateType = templateElem.getAsJsonObject().get(JSON_TEMPLATE_TYPE);
			                    if (!templateType.isJsonNull()) {
				                    String templateTypeId = ((JsonObject) templateType).get(JSON_ID).getAsString();
				                    if (Objects.equals(templateTypeId, id)) {
					                    JsonPrimitive name = templateElem.getAsJsonObject().getAsJsonPrimitive(JSON_NAME);
					                    JsonPrimitive templateId = templateElem.getAsJsonObject().getAsJsonPrimitive(JSON_ID);
					                    if (null != templateId && null != name) {
						                    templateList.add(new Field(Field.Type.TEMPLATE, name.getAsString(), templateId.getAsString()));
					                    }
				                    }
			                    }
                            }
                        }
                        templateGroup.setFields(templateList);
                    }
                }
            }
            templateGroupFieldCache.put(id, templateGroup);
        }
        return templateGroup;
    }

    private List<Field> getTemplateFields(String id) throws IOException {
        List<Field> fields;

        if (templateFieldCache.get(id) != null) {
            fields = templateFieldCache.get(id);
        } else {
            //Getting initial values
            fields = new LinkedList<>();
	        HttpURLConnection connection = getHttpURLConnection(ONEFLOW_BASE_URL
			        + API_HELPERS_URL + API_TEMPLATE_TYPES_URL + id + API_DATA_FIELDS, "GET", true);
	        connection.setRequestProperty("Content-Type", "text/plain");

            int responseCode = connection.getResponseCode();
            if (responseCode == 404) {
                connection.disconnect();
                return null;

            } else if (responseCode > FAILURE_HTTP_RESPONSE_CODE) {
                String error = IOUtils.toString(connection.getErrorStream());
                connection.disconnect();
                log.error(ONEFLOW_BASE_URL + API_DATA_SET_URL + id + ":" + connection.getResponseCode() + connection.getResponseMessage() + "\n" + error);
                throw new IOException(error);

            } else {
                JsonElement agreementsFields = new JsonParser().parse(new InputStreamReader(connection.getInputStream(), StandardCharsets.UTF_8));
                connection.disconnect();
	            if (null != agreementsFields) {
		            JsonArray templateListElem = agreementsFields.getAsJsonObject().get(JSON_DATA_FIELDS).getAsJsonArray();
                    if (null != templateListElem) {
	                    fields.addAll(jsonsTemplateArrayToFields(templateListElem));
                    }
                }
                templateFieldCache.put(id, fields);
            }
        }
        return fields;
    }

    private Document parseDocument(JsonElement documentElement) {
        Document document = null;

        if (documentElement.isJsonObject()) {
            document = new Document();
            JsonObject jsonObject = documentElement.getAsJsonObject();

            JsonPrimitive idElem = jsonObject.getAsJsonPrimitive(JSON_ID);
            if (idElem != null) {
                document.setId(idElem.getAsString());
            }

            JsonPrimitive titleElem = jsonObject.getAsJsonPrimitive(JSON_NAME);
            if (titleElem != null) {
                document.setTitle(titleElem.getAsString());
            }
        }
        return document;
    }

    private HttpURLConnection getHttpURLConnection(String callUrl) throws IOException {
        return getHttpURLConnection(callUrl, "POST");
    }

    private HttpURLConnection getHttpURLConnection(String callUrl, String method) throws IOException {
        return getHttpURLConnection(callUrl, method, false);
    }

    private HttpURLConnection getHttpURLConnection(String callUrl, String method, boolean isSystemCall) throws IOException {
        URL url = new URL(callUrl);
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
	    connection.addRequestProperty("x-oneflow-api-token", SystemProperties.ONEFLOW_SECURITY_TOKEN);
        connection.setRequestMethod(method);
        connection.setDoOutput(true);
        connection.setDoInput(true);
        connection.setInstanceFollowRedirects(false);
        connection.setRequestMethod(method);
        connection.setRequestProperty("charset", "utf-8");
        connection.setUseCaches(false);
        return connection;
    }

    public Map<Document, List<Field>> getElementOptionTemplateFields(int optionId, DatabaseService databaseService) throws IOException {
        Map<Document, List<Field>> optionFields = new HashMap<>();
        Connection connection = null;
        PreparedStatement ps = null;
        ResultSet rs = null;

        try {
            connection = databaseService.getConnection();
            ps = connection.prepareStatement(QUERY_ELEMENT_OPTIONS_TEMPLATE_FIELDS);
            ps.setInt(1, optionId);
            rs = ps.executeQuery();
            Document template;
            while (rs.next()) {
                String templateId = rs.getString(SQL_TEMPLATE_ID);
                String fieldName = rs.getString(SQL_FIELD_NAME);

                if (templateCache.get(templateId) != null) {
                    template = templateCache.get(templateId);
                } else {
                    templateCache.put(templateId, getTemplateGroup(templateId));
                    template = templateCache.get(templateId);
                }
                if (template != null) {
                    List<Field> fields = optionFields.get(template);
                    if (fieldName.matches(REGEX_USER_FIELD)) {
                        if (fields == null) {
                            fields = new ArrayList<>();
                            optionFields.put(template, fields);
                        }
                        fields.add(new Field(Field.Type.USER_FIELD, templateId, fieldName));
                        template.getFields().add(new Field(Field.Type.CHECKBOX, UserField.DELIVERY_CHANNEL.name(), UserField.DELIVERY_CHANNEL.name().toUpperCase()));
                        template.getFields().add(new Field(Field.Type.CHECKBOX, UserField.SIGN_METHOD.name(), UserField.SIGN_METHOD.name().toUpperCase()));
                    } else {
                        for (Field field : template.getFields()) {
                            fields = optionFields.get(template);
                            if (field.getName().equals(fieldName)) {
                                if (fields == null) {
                                    fields = new ArrayList<>();
                                    optionFields.put(template, fields);
                                }
                                fields.add(field);
                            }
                        }
                    }

                }
            }

        } catch (SQLException e) {
            e.printStackTrace();
        } finally {
            DbUtils.closeQuietly(connection, ps, rs);
        }

        return optionFields;
    }

    public Map<Document, List<Field>> getElementTemplateFields(int elementId, DatabaseService databaseService) throws IOException {
        Map<Document, List<Field>> templateFields = new HashMap<>();

        Connection connection = null;
        PreparedStatement ps = null;
        ResultSet rs = null;

        try {
            connection = databaseService.getConnection();
            ps = connection.prepareStatement(QUERY_ELEMENT_TEMPLATE_FIELDS);
            ps.setInt(1, elementId);
            rs = ps.executeQuery();
            Document template;

            while (rs.next()) {
                String templateId = rs.getString(SQL_TEMPLATE_ID);
                String fieldName = rs.getString(SQL_FIELD_NAME);

                if (templateCache.get(templateId) != null) {
                    template = templateCache.get(templateId);
                } else {
                    templateCache.put(templateId, getTemplateGroup(templateId));
                    template = templateCache.get(templateId);
                }
                if (template != null) {
                    List<Field> fields = templateFields.get(template);
                    if (fields == null) {
                        fields = new ArrayList<>();

                    }
                    if (fieldName.matches(REGEX_TEMPLATE_ID)) {
                        List<Field> allFields = template.getFields();
                        List<Field> newFields = getTemplatesInGroupAsFields(templateId).getFields();
                        for (Field newField : newFields) {
                            if (!allFields.contains(newField)) {
                                allFields.add(newField);
                            }
                        }
                        template.setFields(allFields);
                        fields.add(new Field(Field.Type.TEMPLATE, templateId, fieldName));
                    }
                    if (fieldName.matches(REGEX_USER_FIELD)) {
                        Pattern fieldPattern = Pattern.compile(REGEX_USER_DATA_BY_GROUPS);
                        Matcher matcher = fieldPattern.matcher(fieldName);
                        if (matcher.find()) {
                            fields.add(new Field(Field.Type.USER_FIELD, templateId, fieldName));
                        }
                    } else {
                        for (Field field : template.getFields()) {
                            if (field.getName().equals(fieldName)) {
                                fields.add(field);
                            }
                        }
                    }
                    templateFields.put(template, fields);
                }
            }

        } catch (SQLException e) {
            e.printStackTrace();
        } finally {
            DbUtils.closeQuietly(connection, ps, rs);
        }

        return templateFields;
    }

    public List<Field> getTemplateUsers(int surveyId, DatabaseService databaseService) {
        List<Field> templateFields = new ArrayList<>();
        Connection connection = null;
        PreparedStatement ps = null;
        ResultSet rs = null;

        try {
            connection = databaseService.getConnection();
            ps = connection.prepareStatement(QUERY_TEMPLATE_USERS);
            ps.setInt(1, surveyId);
            rs = ps.executeQuery();

            while (rs.next()) {
                String userId = rs.getString(SQL_ID);
                String fieldName = rs.getString(SQL_EL_TEXT);

                if (!StringUtils.isEmpty(userId) && !StringUtils.isEmpty(fieldName)) {
                    templateFields.add(new Field(Field.Type.USER, fieldName, userId));
                }
            }

        } catch (SQLException e) {
            e.printStackTrace();
        } finally {
            DbUtils.closeQuietly(connection, ps, rs);
        }
        return templateFields;
    }

    public String getUserElementIdByMetaId(int surveyId, DatabaseService databaseService) {
        String elId = "";
        Connection connection = null;
        PreparedStatement ps = null;
        ResultSet rs = null;

        try {
            connection = databaseService.getConnection();
            ps = connection.prepareStatement(QUERY_ELEMENT_ID_BY_META_ID);

            ps.setInt(1, surveyId);
            rs = ps.executeQuery();
            while (rs.next()) {
                elId = rs.getString(SQL_ID);
            }
        } catch (SQLException e) {
            e.printStackTrace();
        } finally {
            DbUtils.closeQuietly(connection, ps, rs);
        }
        return elId;
    }

	public String getSignatureFormURL(String backToSiteURL, String templateId, String templateTypeId, Map<String, String> oneflowParams, HttpServletRequest request) throws Exception {
		return getDocumentURI(backToSiteURL, templateId, templateTypeId, oneflowParams, request);
    }

}