package com.imcode.imcms.addon.imsurvey.scrive;

import com.imcode.imcms.addon.imsurvey.FormEngine;
import com.imcode.imcms.api.DatabaseService;
import org.apache.commons.dbutils.DbUtils;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;

public class ElementService {

    /**
     * Assigns Scrive template fields to an element, i.e. a text field, Scrive button or a text area.
     *
     * @param elementId       form engine element id
     * @param templateFields  Scrive template ids mapped to their field names
     * @param databaseService
     */
    public static void assignFieldsToElement(int elementId, Map<String, List<String>> templateFields, DatabaseService databaseService) {
        Connection connection = null;
        PreparedStatement ps = null;

        try {
            connection = databaseService.getConnection();
            connection.setAutoCommit(false);

            ps = connection.prepareStatement("DELETE FROM " + FormEngine.TABLE_PREFIX + "element_scrive_fields WHERE element_id = ?");
            ps.setInt(1, elementId);
            ps.executeUpdate();

            if (templateFields != null && !templateFields.isEmpty()) {
                ps = connection.prepareStatement("INSERT INTO " + FormEngine.TABLE_PREFIX + "element_scrive_fields(element_id, template_id, field_name) VALUES(?, ?, ?)");
                for (String templateId : templateFields.keySet()) {
                    ps.setInt(1, elementId);
                    ps.setString(2, templateId);

                    List<String> fieldNames = templateFields.get(templateId);
                    if (fieldNames != null && !fieldNames.isEmpty()) {
                        for (String fieldName : fieldNames) {
                            ps.setString(3, fieldName);
                            ps.addBatch();
                        }
                    } else {
                        ps.setObject(3, null);
                        ps.addBatch();
                    }
                }

                ps.executeBatch();
            }

            connection.commit();

        } catch (SQLException e) {
            try {
                DbUtils.rollback(connection);
            } catch (SQLException e1) {
                e1.printStackTrace();
            }
            e.printStackTrace();
        } finally {
            DbUtils.closeQuietly(connection, ps, null);
        }
    }

    /**
     * Assigns Scrive template fields to a checkbox or a radio button.
     *
     * @param optionId        a radio button or a checkbox id
     * @param templateFields  Scrive template ids mapped to their field names
     * @param databaseService
     */
    public static void assignFieldsToElementOption(int optionId, Map<String, List<String>> templateFields, DatabaseService databaseService) {
        Connection connection = null;
        PreparedStatement ps = null;

        try {
            connection = databaseService.getConnection();
            connection.setAutoCommit(false);

            ps = connection.prepareStatement("DELETE FROM " + FormEngine.TABLE_PREFIX + "element_option_scrive_fields WHERE option_id = ?");
            ps.setInt(1, optionId);
            ps.executeUpdate();

            if (templateFields != null && !templateFields.isEmpty()) {
                ps = connection.prepareStatement("INSERT INTO " + FormEngine.TABLE_PREFIX + "element_option_scrive_fields(option_id, template_id, field_name) VALUES(?, ?, ?)");
                for (String templateId : templateFields.keySet()) {
                    for (String fieldName : templateFields.get(templateId)) {
                        ps.setInt(1, optionId);
                        ps.setString(2, templateId);
                        ps.setString(3, fieldName);
                        ps.addBatch();
                    }
                }
                ps.executeBatch();
            }

            connection.commit();

        } catch (SQLException e) {
            try {
                DbUtils.rollback(connection);
            } catch (SQLException e1) {
                e1.printStackTrace();
            }
            e.printStackTrace();
        } finally {
            DbUtils.closeQuietly(connection, ps, null);
        }
    }

    /**
     * Removed Scrive fields assigned to the form engine element, and its options.
     *
     * @param elementId
     * @param databaseService
     */
    public static void removeFieldsFromElement(int elementId, DatabaseService databaseService) {
        Connection connection = null;
        PreparedStatement ps = null;

        try {
            connection = databaseService.getConnection();
            connection.setAutoCommit(false);

            ps = connection.prepareStatement("DELETE FROM " + FormEngine.TABLE_PREFIX + "element_option_scrive_fields " +
                    "WHERE option_id IN (SELECT id FROM " + FormEngine.TABLE_PREFIX + "form_elements_options WHERE el_id = ?)");
            ps.setInt(1, elementId);
            ps.executeUpdate();

            ps = connection.prepareStatement("DELETE FROM " + FormEngine.TABLE_PREFIX + "element_scrive_fields WHERE element_id = ?");
            ps.setInt(1, elementId);
            ps.executeUpdate();

            connection.commit();

        } catch (SQLException e) {
            try {
                DbUtils.rollback(connection);
            } catch (SQLException e1) {
                e1.printStackTrace();
            }
            e.printStackTrace();
        } finally {
            DbUtils.closeQuietly(connection, ps, null);
        }
    }

    /**
     * Removes Scrive fields from all form engine elements(and their options) belonging to the survey.
     *
     * @param surveyId
     * @param databaseService
     */
    public static void removeFieldsFromSurvey(int surveyId, DatabaseService databaseService) {
        Connection connection = null;
        PreparedStatement ps = null;
        ResultSet rs = null;

        try {
            connection = databaseService.getConnection();
            ps = connection.prepareStatement("SELECT id FROM " + FormEngine.TABLE_PREFIX + "form_elements WHERE meta_id = ?");
            ps.setInt(1, surveyId);
            rs = ps.executeQuery();
            while (rs.next()) {
                int elementId = rs.getInt(1);
                removeFieldsFromElement(elementId, databaseService);
            }
        } catch (SQLException e) {
            e.printStackTrace();
        } finally {
            DbUtils.closeQuietly(connection, ps, rs);
        }
    }
}
