package com.imcode.imcms.addon.imsurvey.scrive;

import com.google.gson.*;
import com.imcode.imcms.addon.imsurvey.FormEngine;
import com.imcode.imcms.api.DatabaseService;
import org.apache.commons.dbutils.DbUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class ScriveService {
    private static int FAILURE_HTTP_RESPONSE_CODE = 390;

    private String scriveBaseURL = "https://scrive.com";
    private ScriveAuthInfo authInfo;

    private ScriveService() {
    }

    private ScriveService(ScriveAuthInfo authInfo) {
        this.authInfo = authInfo;
    }

    public static ScriveService getInstance(ScriveAuthInfo authInfo) {
        if (authInfo == null ||
                StringUtils.isBlank(authInfo.getConsumerKey()) ||
                StringUtils.isBlank(authInfo.getSignature()) ||
                StringUtils.isBlank(authInfo.getToken()) ||
                StringUtils.isBlank(authInfo.getSecretToken())) {
            return null;
        }

        return new ScriveService(authInfo);
    }

    public boolean isSurveyInSyncWithScrive(int surveyId, DatabaseService databaseService) {
        Map<String, List<String>> surveyFields = getSurveyFields(surveyId, databaseService);

        for (String templateId : surveyFields.keySet()) {
            Document template;
            try {
                template = getDocument(templateId);
            } catch (IOException e) {
                e.printStackTrace();
                return false;
            }

            if (template != null && !template.isDeleted()) {
                List<String> fields = surveyFields.get(template.getId());
                List<String> missingFields = getMissingFields(template, fields);
                if (!missingFields.isEmpty()) {
                    return false;
                }
            } else {
                return false;
            }
        }

        return true;
    }

    public List<String> getMissingFields(Document document, List<String> fieldNames) {
        List<String> missingFields = new ArrayList<String>();

        for (String fieldName : fieldNames) {
            boolean hasField = false;
            for (Field field : document.getFields()) {
                if (field.getName().equals(fieldName)) {
                    hasField = true;
                }
            }

            if (!hasField) {
                missingFields.add(fieldName);
            }
        }

        return missingFields;
    }

    public Map<String, List<String>> getSurveyFields(int surveyId, DatabaseService databaseService) {
        Map<String, List<String>> surveyFields = new HashMap<String, List<String>>();

        Connection connection = null;
        PreparedStatement ps = null;
        ResultSet rs = null;

        try {
            connection = databaseService.getConnection();
            ps = connection.prepareStatement(
                    "SELECT COALESCE(el_fields.template_id, opt_field.template_id) AS template_id, COALESCE(el_fields.field_name, opt_field.field_name) AS field_name \n" +
                            "FROM " + FormEngine.TABLE_PREFIX + "form_elements el \n" +
                            "LEFT JOIN " + FormEngine.TABLE_PREFIX + "form_elements_options opt ON el.id = opt.el_id \n" +
                            "LEFT JOIN " + FormEngine.TABLE_PREFIX + "element_scrive_fields el_fields ON el_fields.`element_id` = el.id \n" +
                            "LEFT JOIN " + FormEngine.TABLE_PREFIX + "element_option_scrive_fields opt_field ON opt_field.`option_id` = opt.id \n" +
                            "WHERE el.meta_id = ?");
            ps.setInt(1, surveyId);
            rs = ps.executeQuery();

            while (rs.next()) {
                String templateId = rs.getString("template_id");
                String fieldName = rs.getString("field_name");

                if (!StringUtils.isEmpty(templateId) && !StringUtils.isEmpty(fieldName)) {
                    List<String> templateFields = surveyFields.get(templateId);
                    if (templateFields == null) {
                        templateFields = new ArrayList<String>();
                        surveyFields.put(templateId, templateFields);
                    }

                    templateFields.add(fieldName);
                }
            }

        } catch (SQLException e) {
            e.printStackTrace();
        } finally {
            DbUtils.closeQuietly(connection, ps, rs);
        }

        return surveyFields;
    }

    private String getDocumentURI(String signSuccessUrl, String templateId, Map<String, String> props) throws Exception {

        int nbrPropsToSet = props.size();

        if (templateId == null || templateId.length() <= 4) {
            throw new IllegalArgumentException("Scrive template id is either missing or invalid: " + templateId);
        }

        HttpURLConnection connection = getHttpURLConnection(getScriveBaseURL() + "/api/v1/createfromtemplate/" + templateId, "POST");
        connection.setRequestProperty("Content-Type", "text/plain");

        int responseCode = connection.getResponseCode();

        boolean parserError = false;
        long docId = 0;
        JsonObject documentToUpdate = null;

        if (responseCode > FAILURE_HTTP_RESPONSE_CODE) {
            String errorMessage = IOUtils.toString(connection.getErrorStream());
            connection.disconnect();
            throw new IOException(errorMessage);
        } else {
            JsonElement parsedReturn = new JsonParser().parse(new InputStreamReader(connection.getInputStream(), "UTF-8"));
            connection.disconnect();
            if (parsedReturn != null && parsedReturn.isJsonObject()) {
                documentToUpdate = parsedReturn.getAsJsonObject();

                JsonPrimitive id = documentToUpdate.getAsJsonPrimitive("id");
                docId = id.getAsLong();
                changeDelivery(documentToUpdate, "mixed");

                JsonElement signaturesTest = documentToUpdate.get("signatories");
                if (null != signaturesTest && signaturesTest.isJsonArray()) {
                    JsonArray signatures = signaturesTest.getAsJsonArray();
                    if (signatures.size() == 2) {
                        /* Use the second signatory for form data. The first one is the author */
                        JsonObject sign2 = signatures.get(1).getAsJsonObject();
                        changeDelivery(sign2, "api");
                        changeProperty("signsuccessredirect", sign2, signSuccessUrl);

                        JsonElement fieldsTest = sign2.get("fields");
                        if (null != fieldsTest && fieldsTest.isJsonArray()) {
                            JsonArray fields = fieldsTest.getAsJsonArray();
                            int valuesSet = 0;

                            for (int i = 0; i < fields.size(); i++) {
                                JsonElement elem2 = fields.get(i);
                                if (null != elem2 && elem2.isJsonObject()) {
                                    JsonObject field = elem2.getAsJsonObject();
                                    JsonElement elemName = field.get("name");
                                    JsonElement elemValue = field.get("value");
                                    JsonElement elemType = field.get("type");
                                    boolean hasPlacement = hasPlacement(field);

                                    if (null != elemName && null != elemValue && null != elemType &&
                                            elemName.isJsonPrimitive() && elemValue.isJsonPrimitive() && elemType.isJsonPrimitive() &&
                                            hasPlacement) {

                                        String fieldName = elemName.getAsString();
                                        if (props.containsKey(fieldName)) {
                                            boolean isCheckbox = "checkbox".equalsIgnoreCase(elemType.getAsString());
                                            String newValue = StringUtils.defaultString(props.get(fieldName));
                                            if (isCheckbox && !newValue.isEmpty()) {
                                                newValue = "true";
                                            }

                                            changeValue(field, newValue);
                                            valuesSet++;
                                        }
                                    }
                                }
                            }
                            if (nbrPropsToSet > 0 && valuesSet != nbrPropsToSet) {
                                parserError = true;
                            }
                        } else {
                            parserError = true;
                        }

                    } else {
                        parserError = true;
                    }
                }

            } else {
                parserError = true;
            }
        }

        if (parserError) {
            throw new Exception("Error template has changed! Can't create document.");
        } else {
            // Update the document with form data
            if (docId > 0) {
                String urlParameters = "json=" + URLEncoder.encode(documentToUpdate.toString(), "UTF-8");

                connection = getHttpURLConnection(getScriveBaseURL() + "/api/v1/update/" + docId);
                connection.setRequestProperty("Content-Type", "application/x-www-form-urlencoded");
                connection.setRequestProperty("Content-Length", "" + Integer.toString(urlParameters.getBytes().length));

                OutputStreamWriter streamWriter = new OutputStreamWriter(connection.getOutputStream());
                streamWriter.write(urlParameters);
                streamWriter.close();

                responseCode = connection.getResponseCode();

                if (responseCode > FAILURE_HTTP_RESPONSE_CODE) {
                    throw new IOException(IOUtils.toString(connection.getErrorStream()));
                }
                connection.disconnect();

                // Ready the document for signing
                connection = getHttpURLConnection(getScriveBaseURL() + "/api/v1/ready/" + docId);
                responseCode = connection.getResponseCode();
                if (responseCode > FAILURE_HTTP_RESPONSE_CODE) {
                    throw new IOException(IOUtils.toString(connection.getErrorStream()));
                } else {
                    /* Get signature page link */
                    JsonElement parsedReturn = new JsonParser().parse(new InputStreamReader(connection.getInputStream(), "UTF-8"));
                    connection.disconnect();
                    if (parsedReturn.isJsonObject()) {
                        documentToUpdate = parsedReturn.getAsJsonObject();

                        JsonElement signaturesTest = documentToUpdate.get("signatories");
                        if (null != signaturesTest && signaturesTest.isJsonArray()) {
                            JsonArray signatures = signaturesTest.getAsJsonArray();
                            if (signatures.size() == 2) {
                                JsonObject sign2 = signatures.get(1).getAsJsonObject();

                                JsonPrimitive signLinkElement = sign2.getAsJsonPrimitive("signlink");
                                if (null != signLinkElement) {
                                    return signLinkElement.getAsString();
                                }
                            } else {
                                throw new Exception("Error not twp signing posts");
                            }
                        }
                    }
                }
                connection.disconnect();
            }
        }

        return null;
    }

    public List<Document> getTemplates() throws IOException {
        List<Document> templates = new ArrayList<Document>();

        HttpURLConnection connection = getHttpURLConnection(getScriveBaseURL() + "/api/v1/list?documentType=Template&limit=1000", "GET");
        if (connection.getResponseCode() > FAILURE_HTTP_RESPONSE_CODE) {
            String error = IOUtils.toString(connection.getErrorStream());
            connection.disconnect();
            throw new IOException(error);
        } else {
            JsonElement parsedReturn = new JsonParser().parse(new InputStreamReader(connection.getInputStream(), "UTF-8"));
            connection.disconnect();
            if (parsedReturn.isJsonObject()) {
                JsonElement templateListElem = ((JsonObject) parsedReturn).get("list");
                if (null != templateListElem && templateListElem.isJsonArray()) {
                    for (JsonElement templateElem : ((JsonArray) templateListElem)) {

                        if (null != templateElem && templateElem.isJsonObject()) {
                            JsonObject templateObj = (JsonObject) templateElem;

                            JsonElement templateFieldElem = templateObj.get("fields");
                            if (null != templateFieldElem && templateFieldElem.isJsonObject()) {

                                JsonObject templateFieldObj = (JsonObject) templateFieldElem;
                                JsonPrimitive id = templateFieldObj.getAsJsonPrimitive("id");
                                JsonPrimitive title = templateFieldObj.getAsJsonPrimitive("title");
                                if (null != id && null != title) {
                                    Document template = getDocument(id.getAsString());
                                    if (template != null) {
                                        templates.add(template);
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        return templates;
    }

    public Document getDocument(String id) throws IOException {
        Document document = null;

        HttpURLConnection connection = getHttpURLConnection(getScriveBaseURL() + "/api/v1/get/" + id, "GET");
        connection.setRequestProperty("Content-Type", "text/plain");

        int responseCode = connection.getResponseCode();
        if (responseCode == 404) {
            connection.disconnect();
            return null;

        } else if (responseCode > FAILURE_HTTP_RESPONSE_CODE) {
            String error = IOUtils.toString(connection.getErrorStream());
            connection.disconnect();
            throw new IOException(error);

        } else {
            JsonElement parsedReturn = new JsonParser().parse(new InputStreamReader(connection.getInputStream(), "UTF-8"));
            connection.disconnect();
            if (parsedReturn.isJsonObject()) {
                document = parseDocument(parsedReturn);
            }
        }

        return document;
    }

    private Document parseDocument(JsonElement documentElement) {
        Document document = null;

        if (documentElement.isJsonObject()) {
            document = new Document();
            JsonObject jsonObject = documentElement.getAsJsonObject();

            JsonElement idElem = jsonObject.get("id");
            if (idElem != null && idElem.isJsonPrimitive()) {
                document.setId(idElem.getAsString());
            }

            JsonElement titleElem = jsonObject.get("title");
            if (titleElem != null && titleElem.isJsonPrimitive()) {
                document.setTitle(titleElem.getAsString());
            }

            boolean deleted = false;
            JsonElement deletedElem = jsonObject.get("deleted");
            if (deletedElem != null && deletedElem.isJsonPrimitive()) {
                deleted = deletedElem.getAsBoolean();
            }
            document.setDeleted(deleted);

            JsonElement signaturesTest = jsonObject.get("signatories");
            if (null != signaturesTest && signaturesTest.isJsonArray()) {
                JsonArray signatures = signaturesTest.getAsJsonArray();
                // TODO: Check if signatory is not an author
                if (signatures.size() == 2) {
                    // Custom signatory
                    JsonObject sign2 = signatures.get(1).getAsJsonObject();

                    JsonElement fieldsTest = sign2.get("fields");
                    if (null != fieldsTest && fieldsTest.isJsonArray()) {
                        JsonArray fields = fieldsTest.getAsJsonArray();
                        for (int i = 0; i < fields.size(); i++) {
                            JsonElement elem2 = fields.get(i);
                            if (null != elem2 && elem2.isJsonObject()) {
                                JsonObject field = elem2.getAsJsonObject();
                                JsonElement elemName = field.get("name");
                                JsonElement elemType = field.get("type");
                                boolean hasPlacement = hasPlacement(field);

                                if (null != elemName && null != elemType && elemName.isJsonPrimitive() && elemType.isJsonPrimitive() && hasPlacement) {
                                    boolean isCheckbox = "checkbox".equalsIgnoreCase(elemType.getAsString());

                                    String fieldName = elemName.getAsString();
                                    document.getFields().add(new Field(isCheckbox ? Field.Type.CHECKBOX : Field.Type.TEXT, fieldName));
                                }
                            }
                        }
                    }
                }
            }
        }
        return document;
    }

    public static boolean hasPlacement(JsonObject fieldElem) {
        JsonElement elemPlacement = fieldElem.get("placements");
        return elemPlacement != null && elemPlacement.isJsonArray() && elemPlacement.getAsJsonArray().size() > 0;
    }

    private HttpURLConnection getHttpURLConnection(String callUrl) throws IOException {
        return getHttpURLConnection(callUrl, "POST");
    }

    private HttpURLConnection getHttpURLConnection(String callUrl, String method) throws IOException {
        String oauth_consumer_key = getAuthInfo().getConsumerKey();
        String oauth_signature = getAuthInfo().getSignature();
        String oauth_token = getAuthInfo().getToken();
        String oauth_token_secret = getAuthInfo().getSecretToken();

        StringBuilder authHead = new StringBuilder();
        authHead.append("OAuth realm=\"Scrive\", ");
        //creadHead.append("oauth_callback=\"").append(callbackUrl).append("\", ");
        authHead.append("oauth_signature_method=\"PLAINTEXT\", ");
        authHead.append("oauth_consumer_key=\"").append(oauth_consumer_key).append("\", ");
        authHead.append("oauth_token=\"").append(oauth_token).append("\", ");
        authHead.append("oauth_signature=\"").append(oauth_signature).append("&").append(oauth_token_secret).append("\" ");

        URL url = new URL(callUrl);
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        connection.setDoOutput(true);
        connection.setDoInput(true);
        connection.setInstanceFollowRedirects(false);
        connection.setRequestMethod(method);
        connection.setRequestProperty("Authorization", authHead.toString());
        connection.setRequestProperty("Charset", "utf-8");
        connection.setUseCaches(false);
        return connection;
    }

    private void changeValue(JsonObject field, String data) {
        changeProperty("value", field, data);
    }

    private void changeDelivery(JsonObject parent, String data) {
        changeProperty("delivery", parent, data);
    }

    private void changeProperty(String property, JsonObject parent, String data) {
        parent.remove(property);
        parent.addProperty(property, data);
    }

    public Map<Document, List<Field>> getElementOptionTemplateFields(int optionId, DatabaseService databaseService) throws IOException {
        Map<Document, List<Field>> optionFields = new HashMap<Document, List<Field>>();
        Connection connection = null;
        PreparedStatement ps = null;
        ResultSet rs = null;

        try {
            connection = databaseService.getConnection();
            ps = connection.prepareStatement("SELECT * FROM " + FormEngine.TABLE_PREFIX + "element_option_scrive_fields WHERE option_id = ?");
            ps.setInt(1, optionId);
            rs = ps.executeQuery();
            while (rs.next()) {
                String templateId = rs.getString("template_id");
                String fieldName = rs.getString("field_name");
                Document template = getDocument(templateId);
                if (template != null) {
                    for (Field field : template.getFields()) {
                        if (field.getName().equals(fieldName)) {
                            List<Field> fields = optionFields.get(template);
                            if (fields == null) {
                                fields = new ArrayList<Field>();
                                optionFields.put(template, fields);
                            }

                            fields.add(field);
                        }
                    }
                }
            }

        } catch (SQLException e) {
            e.printStackTrace();
        } finally {
            DbUtils.closeQuietly(connection, ps, rs);
        }

        return optionFields;
    }

    public Map<Document, List<Field>> getElementTemplateFields(int elementId, DatabaseService databaseService) throws IOException {
        Map<Document, List<Field>> templateFields = new HashMap<Document, List<Field>>();

        Connection connection = null;
        PreparedStatement ps = null;
        ResultSet rs = null;

        try {
            connection = databaseService.getConnection();
            ps = connection.prepareStatement("SELECT * FROM " + FormEngine.TABLE_PREFIX + "element_scrive_fields WHERE element_id = ?");
            ps.setInt(1, elementId);
            rs = ps.executeQuery();
            while (rs.next()) {
                String templateId = rs.getString("template_id");
                String fieldName = rs.getString("field_name");
                Document template = getDocument(templateId);
                if (template != null) {
                    List<Field> fields = templateFields.get(template);
                    if (fields == null) {
                        fields = new ArrayList<Field>();
                        templateFields.put(template, fields);
                    }

                    for (Field field : template.getFields()) {
                        if (field.getName().equals(fieldName)) {
                            fields.add(field);
                        }
                    }
                }
            }

        } catch (SQLException e) {
            e.printStackTrace();
        } finally {
            DbUtils.closeQuietly(connection, ps, rs);
        }

        return templateFields;
    }

    public String getSignatureFormURL(String backToSiteURL, String templateId, Map<String, String> scriveParams) throws Exception {
        String scringDocumentURI = getDocumentURI(backToSiteURL, templateId, scriveParams);
        if (scringDocumentURI == null) {
            return null;
        }

        return getScriveBaseURL() + scringDocumentURI;
    }

    public String getScriveBaseURL() {
        return scriveBaseURL;
    }

    public void setScriveBaseURL(String scriveBaseURL) {
        this.scriveBaseURL = scriveBaseURL;
    }

    public ScriveAuthInfo getAuthInfo() {
        return authInfo;
    }

}
