package com.imcode.imcms.addon.imagearchive.service.jpa;

import com.imcode.imcms.addon.imagearchive.Config;
import com.imcode.imcms.addon.imagearchive.command.SearchImageCommand;
import com.imcode.imcms.addon.imagearchive.entity.Exif;
import com.imcode.imcms.addon.imagearchive.entity.Image;
import com.imcode.imcms.addon.imagearchive.entity.Keyword;
import com.imcode.imcms.addon.imagearchive.repository.ImageRepository;
import com.imcode.imcms.addon.imagearchive.service.Facade;
import com.imcode.imcms.addon.imagearchive.service.ImageService;
import com.imcode.imcms.addon.imagearchive.service.file.FileServiceImpl;
import com.imcode.imcms.addon.imagearchive.util.PageResult;
import com.imcode.imcms.addon.imagearchive.util.Pagination;
import com.imcode.imcms.addon.imagearchive.util.exif.ExifData;
import com.imcode.imcms.addon.imagearchive.util.exif.ExifUtils;
import com.imcode.imcms.addon.imagearchive.util.exif.Flash;
import com.imcode.imcms.addon.imagearchive.util.image.ImageInfo;
import com.imcode.imcms.addon.imagearchive.util.image.ImageOp;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.ejb.criteria.OrderImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.orm.jpa.JpaTransactionManager;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.EntityManager;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.*;
import java.io.*;
import java.util.*;
import java.util.regex.Matcher;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import static com.imcode.imcms.addon.imagearchive.command.SearchImageCommand.*;

@Service
@Transactional
public class ImageServiceImpl implements ImageService {
    private static final Log log = LogFactory.getLog(ImageServiceImpl.class);

    @Autowired
    JpaTransactionManager transactionManager;

    @Autowired
    private Facade facade;
    @Autowired
    private ExifServiceImpl exifService;
    @Autowired
    private ImageRepository imageRepository;
    @Autowired
    private KeywordServiceImpl keywordsService;

    @Autowired
    private Config config;

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public Image findById(final Long imageId) {
        final Image image = imageRepository.findOne(imageId);

        if (image != null) {
            final Exif exif = exifService.getById(imageId);
            image.setExif(exif);
        }

        return image;
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public Exif findExifByPK(Long imageId) {
        Image image = imageRepository.findByIDAndFetchExif(imageId);

        if (image != null) {
            return exifService.getById(imageId);
        }

        return null;
    }

    public Image createImage(File tempFile, ImageInfo imageInfo, String imageName, short status) {
        Image image = create(tempFile, imageInfo, imageName);
        image.setStatus(status);

        image = imageRepository.save(image);

        if (!facade.getFileService().storeImage(tempFile, image.getId(), false)) {
            return null;
        }

        return image;
    }

    private Image create(File tempFile, ImageInfo imageInfo, String imageName) {
        Image image = new Image();

        String copyright = "";
        String description = "";
        String artist = "";
        String manufacturer = null;
        String model = null;
        String compression = null;
        Double exposure = null;
        String exposureProgram = null;
        Float fStop = null;
        Date dateOriginal = null;
        Date dateDigitized = null;
        Flash flash = null;
        Float focalLength = null;
        String colorSpace = null;
        Integer xResolution = null;
        Integer yResolution = null;
        Integer resolutionUnit = null;
        Integer pixelXDimension = null;
        Integer pixelYDimension = null;
        Integer ISO = null;

        ExifData data = ExifUtils.getExifData(tempFile);
        if (data != null) {
            copyright = StringUtils.substring(data.getCopyright(), 0, 255);
            description = StringUtils.substring(data.getDescription(), 0, 255);
            artist = StringUtils.substring(data.getArtist(), 0, 255);
            xResolution = data.getxResolution();
            yResolution = data.getyResolution();
            manufacturer = data.getManufacturer();
            model = data.getModel();
            compression = data.getCompression();
            exposure = data.getExposure();
            exposureProgram = data.getExposureProgram();
            fStop = data.getfStop();
            flash = data.getFlash();
            focalLength = data.getFocalLength();
            colorSpace = data.getColorSpace();
            resolutionUnit = data.getResolutionUnit();
            pixelXDimension = data.getPixelXDimension();
            pixelYDimension = data.getPixelYDimension();
            dateOriginal = data.getDateOriginal();
            dateDigitized = data.getDateDigitized();
            ISO = data.getISO();
        }

        Exif originalExif = new Exif(xResolution, yResolution, description, artist, copyright,
                manufacturer, model, compression, exposure, exposureProgram, fStop, flash, focalLength, colorSpace,
                resolutionUnit, pixelXDimension, pixelYDimension, dateOriginal, dateDigitized, ISO);

        String uploadedBy = "";
        image.setUploadedBy(uploadedBy);

        image.setName(StringUtils.substring(imageName, 0, 255));
        image.setFormat(imageInfo.getFormat().getOrdinal());
        image.setFileSize((int) tempFile.length());
        image.setWidth(imageInfo.getWidth());
        image.setHeight(imageInfo.getHeight());
        image.setExif(originalExif);
        return image;
    }

    public List<Image> createImagesFromZip(File tempFile) {
        ZipFile zip = null;
        List<Image> createdImages = new ArrayList<Image>();

        try {
            zip = new ZipFile(tempFile, ZipFile.OPEN_READ);

            Enumeration<? extends ZipEntry> entries = zip.entries();
            while (entries.hasMoreElements()) {
                ZipEntry entry = entries.nextElement();

                String fileName = entry.getName();
                Matcher matcher = FileServiceImpl.FILENAME_PATTERN.matcher(fileName);

                if (fileName.startsWith(FileServiceImpl.OSX_RESOURCE_FORK_PREFIX) || !matcher.matches() || StringUtils.isEmpty((fileName = matcher.group(1).trim()))) {
                    continue;
                }

                String extension = StringUtils.substringAfterLast(fileName, ".").toLowerCase();
                if (!FileServiceImpl.IMAGE_EXTENSIONS_SET.contains(extension)) {
                    continue;
                }

                File entryFile = facade.getFileService().createTemporaryFile("zipEntryTmp");

                InputStream inputStream = null;
                OutputStream outputStream = null;
                try {
                    inputStream = zip.getInputStream(entry);
                    outputStream = new BufferedOutputStream(FileUtils.openOutputStream(entryFile));

                    IOUtils.copy(inputStream, outputStream);
                    outputStream.flush();
                    IOUtils.closeQuietly(outputStream);
                    IOUtils.closeQuietly(inputStream);
                    ImageInfo imageInfo = ImageOp.getImageInfo(config, entryFile);
                    if (imageInfo == null || imageInfo.getFormat() == null
                            || imageInfo.getWidth() < 1 || imageInfo.getHeight() < 1) {
                        continue;
                    }
                    Image image = this.createImage(entryFile, imageInfo, fileName, Image.STATUS_ACTIVE);
                    if (image != null) {
                        createdImages.add(image);
                    }

                } catch (Exception ex) {
                    log.warn(ex.getMessage(), ex);
                    entryFile.delete();
                } finally {
                    IOUtils.closeQuietly(outputStream);
                    IOUtils.closeQuietly(inputStream);
                    entryFile.delete();
                }
            }

        } catch (Exception ex) {
            log.warn(ex.getMessage(), ex);
        } finally {
            if (zip != null) {
                try {
                    zip.close();
                } catch (IOException ex) {
                    log.warn(ex.getMessage(), ex);
                }
            }
        }

        return createdImages;
    }

    public void createImages(List<Object[]> tuples) {
        for (Object[] tuple : tuples) {
            File tempFile = (File) tuple[0];
            ImageInfo imageInfo = (ImageInfo) tuple[1];
            String imageName = (String) tuple[2];
            createImage(tempFile, imageInfo, imageName, Image.STATUS_ACTIVE);
        }
    }

    public void deleteImage(Long imageId) {
        imageRepository.delete(imageId);

        facade.getFileService().deleteImage(imageId);
    }

    public void updateFullData(final Image image, final List<Long> categoryIds, final List<String> imageKeywords) {
        image.setCategories(categoryIds);
        updateImageKeywordsWithoutSave(image, imageKeywords);
        imageRepository.save(image);
    }

    public void updateData(final Image image, final List<Long> categoryIds, final List<String> imageKeywords) {
        updateFullData(image, categoryIds, imageKeywords);
    }

    private void updateImageCategories(Image image, List<Long> categoryIds) {
        image.setCategories(categoryIds);
        imageRepository.save(image);
    }

    private void updateImageKeywords(Image image, List<String> imageKeywords) {
        updateImageKeywordsWithoutSave(image, imageKeywords);
        imageRepository.save(image);
    }

    private Image updateImageKeywordsWithoutSave(Image image, List<String> imageKeywords) {
        List<Keyword> keywords = new LinkedList<Keyword>();

        for (String keywordString : imageKeywords) {
            Keyword keyword = keywordsService.createOrGet(keywordString);
            keywords.add(keyword);
        }

        image.setKeywords(keywords);
        return image;
    }

    public void archiveImage(final Long imageId) {
        setStatus(imageId, Image.STATUS_ARCHIVED);
    }

    private void setStatus(Long imageId, short status) {
        Image image = imageRepository.findOne(imageId);
        if (image != null) {
            image.setStatus(status);
            imageRepository.save(image);
        }
    }

    public void unarchiveImage(final Long imageId) {
        setStatus(imageId, Image.STATUS_ACTIVE);
    }

    public List<Image> getAllImages() {
        return imageRepository.findAll();
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public PageResult<Image> searchImages(final Pagination pag, SearchImageCommand command) {

        EntityManager entityManager = transactionManager.getEntityManagerFactory().createEntityManager();
        CriteriaBuilder builder = entityManager.getCriteriaBuilder();

        CriteriaQuery<Image> criteriaQuery = builder.createQuery(Image.class);

        Root<Image> root = criteriaQuery.from(Image.class);
        Join<Image, Keyword> keywordsJoin = root.join("keywords", JoinType.LEFT);
        Join<Image, Exif> exifJoin = root.join("exif", JoinType.LEFT);
        Predicate whereClause = null;

        if (command.getKeywordId() > 0) {
            whereClause = builder.equal(keywordsJoin.<Long>get("id"), command.getKeywordId());
        }
        if (StringUtils.isNotBlank(command.getArtist())) {
            Predicate artistPredicate = builder.equal(exifJoin.<String>get("artist"), command.getArtist());
            whereClause = andClauses(whereClause, artistPredicate, builder);
        }

        whereClause = getFreeTextClause(command, whereClause, root, exifJoin, builder);
        whereClause = getShowingClause(command, whereClause, root, builder);
        criteriaQuery.where(whereClause);

        final Order ordering = resolveOrdering(command, root);
        criteriaQuery.orderBy(ordering);

        final TypedQuery<Image> query = entityManager.createQuery(criteriaQuery.select(root).distinct(true));

        List<Image> images = query.getResultList();
        images = filterImagesByCategories(images, command);
        pag.update(images.size());

        if (command.getSortBy() == SORT_BY_ARTIST) {
            images = sortByArtist(images, command.getSortOrder());
        }

        images = applyPaging(images, pag);

        return new PageResult<Image>(images, pag);
    }

    private List<Image> sortByArtist(List<Image> images, short sortOrder) {
        for (Image image : images) {
            // restoring exif
            if (image.getExif() == null) {
                final Exif exif = exifService.getById(image.getId());
                image.setExif(exif);
            }
        }

        final boolean descending = BooleanUtils.toBoolean(sortOrder);
        final int orderCoefficient = (descending) ? 1 : -1;

        Collections.sort(images, new Comparator<Image>() {
            @Override
            public int compare(Image image1, Image image2) {
                try {
                    final Exif exif1 = image1.getExif();
                    final Exif exif2 = image2.getExif();

                    final String artist1 = (exif1 == null) ? "" : exif1.getArtist();
                    final String artist2 = (exif2 == null) ? "" : exif2.getArtist();

                    final int result;

                    if (artist1.equals(artist2)) {
                        result = 0;

                    } else if ("".equals(artist1)) {
                        result = 1;

                    } else if ("".equals(artist2)) {
                        result = -1;

                    } else {
                        result = artist1.compareTo(artist2);
                    }

                    return result * orderCoefficient;
                } catch (Exception e) {
                    return 0;
                }
            }
        });

        return images;
    }

    private List<Image> applyPaging(List<Image> images, Pagination pagination) {
        final List<Image> pagedImages = new ArrayList<Image>();
        final int startPosition = pagination.getStartPosition();
        final int currentPage = pagination.getCurrentPage();
        final int pageSize = pagination.getPageSize();
        final int endPosition = (currentPage + 1) * pageSize;

        for (int i = 0; i < images.size(); i++) {
            if ((i < startPosition) || (i >= endPosition)) {
                continue;
            }

            pagedImages.add(images.get(i));
        }

        return pagedImages;
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    private Predicate getFreeTextClause(SearchImageCommand command, Predicate existingPredicate, Root<Image> root,
                                        Join<Image, Exif> exifJoin, CriteriaBuilder builder) {

        if (StringUtils.isNotBlank(command.getFreetext())) {
            String formattedFreeText = "%" + command.getFreetext().trim() + "%";
            Predicate likePredicate = builder.like(root.<String>get("name"), formattedFreeText);

            if (!command.isFileNamesOnly()) {
                likePredicate = builder.or(likePredicate,
                        builder.like(root.<String>get("altText"), formattedFreeText),
                        builder.like(exifJoin.<String>get("description"), formattedFreeText),
                        builder.like(exifJoin.<String>get("artist"), formattedFreeText),
                        builder.like(exifJoin.<String>get("copyright"), formattedFreeText),
                        builder.like(exifJoin.<String>get("manufacturer"), formattedFreeText),
                        builder.like(exifJoin.<String>get("model"), formattedFreeText),
                        builder.like(exifJoin.<String>get("compression"), formattedFreeText),
                        builder.like(exifJoin.<String>get("colorSpace"), formattedFreeText),
                        builder.like(exifJoin.<String>get("colorSpace"), formattedFreeText));
            }

            existingPredicate = andClauses(existingPredicate, likePredicate, builder);
        }

        return existingPredicate;
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    private Predicate getShowingClause(SearchImageCommand command, Predicate existingPredicate, Root<Image> root,
                                       CriteriaBuilder builder) {
        boolean showDeleted = false;

        if (command.getShow() != SearchImageCommand.SHOW_ALL) {
            Predicate showProperty;
            switch (command.getShow()) {
                case SearchImageCommand.SHOW_ERASED: {
                    showProperty = builder.equal(root.<Short>get("status"), Image.STATUS_ARCHIVED);
                    showDeleted = true;
                    break;
                }
                case SearchImageCommand.SHOW_WITH_VALID_LICENCE: {
                    // even though corresponding radiobutton is hidden and this case not active
                    Predicate licensedBefore = builder.lessThanOrEqualTo(root.<Date>get("licenseDt"), new Date());
                    Predicate licenseEndsLater = builder.greaterThanOrEqualTo(root.<Date>get("licenseEndDt"), new Date());
                    showProperty = builder.and(licensedBefore, licenseEndsLater);
                    break;
                }
                case SearchImageCommand.SHOW_NEW:
                default: {
                    Calendar cal = new GregorianCalendar();
                    cal.setTime(new Date());
                    cal.set(Calendar.HOUR_OF_DAY, 0);
                    cal.set(Calendar.MINUTE, 0);
                    cal.set(Calendar.SECOND, 0);
                    cal.set(Calendar.MILLISECOND, 0);
                    Date today = cal.getTime();
                    showProperty = builder.greaterThanOrEqualTo(root.<Date>get("createdDt"), today);
                    break;
                }
            }

            existingPredicate = andClauses(existingPredicate, showProperty, builder);
        }

        if (!showDeleted) {
            Predicate showProperty = builder.notEqual(root.<Short>get("status"), Image.STATUS_ARCHIVED);
            existingPredicate = andClauses(existingPredicate, showProperty, builder);
        }

        return existingPredicate;
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    private Predicate andClauses(Predicate nullablePredicate, Predicate notNullPredicate, CriteriaBuilder builder) {
        return (nullablePredicate == null)
                ? notNullPredicate
                : builder.and(nullablePredicate, notNullPredicate);
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    private Order resolveOrdering(SearchImageCommand command, Root<Image> root) {
        Expression<?> sortExpression;
        short sortBy = command.getSortBy();
        boolean ascending = !BooleanUtils.toBoolean(command.getSortOrder());

        switch (sortBy) {
            case SORT_BY_ALPHABET: {
                sortExpression = root.<String>get("name");
                break;
            }
            case SORT_BY_ENTRY_DATE:
            default: {
                sortExpression = root.<Date>get("createdDt");
                break;
            }
        }

        return new OrderImpl(sortExpression, ascending);
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    private List<Image> filterImagesByCategories(List<Image> images, SearchImageCommand command) {
        // CriteriaBuilder not used because there is no any possibility to check is collection contains any element
        // from other collection using Hibernate, only check is element in collection
        List<Long> categoryIds = command.getCategoryIds();
        boolean hasNoCategory = categoryIds.remove(-2L);
        if (hasNoCategory) {
            return (categoryIds.size() > 0)
                    ? containsOrHasEmptyCategories(images, categoryIds)
                    : hasEmptyCategories(images);
        } else {
            return containsCategories(images, categoryIds);
        }
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    private List<Image> containsCategories(List<Image> images, final List<Long> categoryIds) {
        CollectionUtils.filter(images, new org.apache.commons.collections4.Predicate<Image>() {
            @Override
            public boolean evaluate(Image image) {
                final List<Long> categories = image.getCategories();
                return CollectionUtils.containsAny(categories, categoryIds);
            }
        });

        return images;
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    private List<Image> hasEmptyCategories(List<Image> images) {
        CollectionUtils.filter(images, new org.apache.commons.collections4.Predicate<Image>() {
            @Override
            public boolean evaluate(Image image) {
                final List<Long> categories = image.getCategories();
                return (categories.size() == 0);
            }
        });

        return images;
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    private List<Image> containsOrHasEmptyCategories(List<Image> images, final List<Long> categoryIds) {
        CollectionUtils.filter(images, new org.apache.commons.collections4.Predicate<Image>() {
            @Override
            public boolean evaluate(Image image) {
                final List<Long> categories = image.getCategories();

                final boolean containsAny = CollectionUtils.containsAny(categories, categoryIds);
                final boolean hasNoCategories = categories.size() == 0;

                return containsAny || hasNoCategories;
            }
        });

        return images;
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public List<Keyword> findAvailableKeywords(final Long imageId) {
        List<Keyword> allKeywords = keywordsService.getKeywords();
        List<Keyword> imageKeywords = findImageKeywords(imageId);
        allKeywords.removeAll(imageKeywords);

        return allKeywords;
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public List<Keyword> findImageKeywords(final Long imageId) {
        Image image = imageRepository.findOne(imageId);
        return image.getKeywords();
    }

    public File uploadFile(byte[] data) {
        File tempFile = facade.getFileService().createTemporaryFile("img_upload_" + Math.round(Math.random() * 100000));
        try {
            FileOutputStream output = new FileOutputStream(tempFile);
            IOUtils.write(data, output);
        } catch (IOException e) {
            return null;
        }
        return tempFile;
    }
}
