package com.imcode.imcms.addon.imagearchive.service.jpa;

import com.imcode.imcms.addon.imagearchive.entity.Keyword;
import com.imcode.imcms.addon.imagearchive.repository.KeywordsRepository;
import com.imcode.imcms.addon.imagearchive.service.KeywordService;
import com.imcode.imcms.addon.imagearchive.service.exception.KeywordExistsException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;

@Service
@Transactional
public class KeywordServiceImpl implements KeywordService {
    @Autowired
    private KeywordsRepository keywordsRepository;

    public Keyword createKeyword(final String newKeywordName) throws KeywordExistsException {
        if (existsKeyword(newKeywordName)) {
            throw new KeywordExistsException();
        }

        Keyword keyword = new Keyword();
        keyword.setName(newKeywordName);
        keywordsRepository.save(keyword);
        return keyword;
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public List<Keyword> getKeywords() {
        List<Keyword> keywords = keywordsRepository.findAll();
        return keywords;
    }

    @Transactional
    public void deleteKeyword(final Long keywordId) {
        if (!keywordsRepository.exists(keywordId)) return;
        removeKeywordImages(keywordId);
        try {
            keywordsRepository.delete(keywordId);
        } catch (Exception ignore) {
            Keyword keyword = keywordsRepository.getOne(keywordId);
            keywordsRepository.save(keyword);
        } finally {
            try {
                keywordsRepository.flush();
            } catch (Exception ignore) {
                // only now keyword should be deleted
            }
        }
    }

    @Transactional
    private void removeKeywordImages(Long keywordId) {
        Keyword keyword = keywordsRepository.getOne(keywordId);
        keyword.getImages().clear();
        keywordsRepository.save(keyword);
        keywordsRepository.flush();
    }

    public void updateKeyword(final Long keywordId, final String keywordName) throws KeywordExistsException {
        Keyword keyword = keywordsRepository.findOne(keywordId);
        if (keyword != null) {
            keyword.setName(keywordName);
            keywordsRepository.save(keyword);
        }
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public boolean existsKeyword(final String newKeywordName) {
        return keywordsRepository.findByName(newKeywordName) != null;
    }

    public List<Keyword> findKeywordsByNameLike(String text) {
        return keywordsRepository.findByNameLike(text);
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public Keyword getKeyword(Long id) {
        return keywordsRepository.findOne(id);
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public Keyword findByName(String name) {
        return keywordsRepository.findByName(name);
    }

    public Keyword createOrGet(String name) {
        try {
            return createKeyword(name);
        } catch (KeywordExistsException e) {
            return findByName(name);
        }
    }

    public List<Keyword> findByName(List<String> names) {
        List<Keyword> keywords = new ArrayList<Keyword>();

        for (String name : names) {
            Keyword keyword = findByName(name);

            if (keyword != null) {
                keywords.add(keyword);
            }
        }

        return keywords;
    }
}
