package com.imcode.imcms.addon.imagearchive.validator;

import com.imcode.imcms.addon.imagearchive.service.Facade;
import com.imcode.imcms.addon.imagearchive.util.image.ImageInfo;
import com.imcode.imcms.addon.imagearchive.util.image.ImageOp;
import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.validation.Errors;
import org.springframework.validation.Validator;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.commons.CommonsMultipartFile;

import java.io.File;
import java.util.zip.ZipFile;

public class ImageUploadValidator implements Validator {
    private static final Log log = LogFactory.getLog(ImageUploadValidator.class);

    private Facade facade;
    private File tempFile;
    private ImageInfo imageInfo;
    private String imageName;
    private boolean zipFile;


    public ImageUploadValidator(Facade facade) {
        this.facade = facade;
    }


    @SuppressWarnings("unchecked")
    public boolean supports(Class clazz) {
        return MultipartFile.class.isAssignableFrom(clazz);
    }

    public void validate(Object target, Errors errors) {
        MultipartFile file = (MultipartFile) target;
        File localTempFile = null;
        if (file == null || file.isEmpty()) {
            errors.rejectValue("file", "addImage.invalidImageError");

            return;
        }

        try {
            imageName = file.getOriginalFilename();
            tempFile = facade.getImageService().uploadFile(file.getBytes());
            localTempFile = File.createTempFile("" + Math.random() * 1000, ".tmp");
            file.transferTo(localTempFile);
            ZipFile zip = null;
            try {
                zip = new ZipFile(localTempFile, ZipFile.OPEN_READ);
                zipFile = true;
            } catch (Exception ex) {
            } finally {
                if (zip != null) {
                    zip.close();
                }
            }

            long maxZipUploadSize = facade.getConfig().getMaxZipUploadSize();
            long maxImageUploadSize = facade.getConfig().getMaxImageUploadSize();
            long fileLength = localTempFile.length();
            double size = 0.0;
            boolean sizeError = false;

            if (zipFile && fileLength > maxZipUploadSize) {
                sizeError = true;
                size = maxZipUploadSize;
            } else if (!zipFile && fileLength > maxImageUploadSize) {
                sizeError = true;
                size = maxImageUploadSize;
            }

            if (sizeError) {
                size /= (1024.0 * 1024.0);
                errors.rejectValue("file", "addImage.sizeError", new Object[]{size}, "???");

                return;
            }

            if (!zipFile) {
                imageInfo = ImageOp.getImageInfo(facade.getConfig(), localTempFile);
                if (imageInfo == null || imageInfo.getFormat() == null
                        || imageInfo.getWidth() < 1 || imageInfo.getHeight() < 1) {
                    errors.rejectValue("file", "addImage.invalidImageError");
                    localTempFile.delete();
                }
            }
        } catch (Exception ex) {
            errors.rejectValue("file", "addImage.invalidImageError");
        } finally {
            FileUtils.deleteQuietly(localTempFile);
            if (file instanceof CommonsMultipartFile)
                ((CommonsMultipartFile) file).getFileItem().delete();
        }
    }


    public File getTempFile() {
        return tempFile;
    }

    public ImageInfo getImageInfo() {
        return imageInfo;
    }

    public String getImageName() {
        return imageName;
    }

    public boolean isZipFile() {
        return zipFile;
    }
}
