package com.imcode.imcms.addon.imagearchive.service.jpa;

import com.imcode.imcms.addon.imagearchive.Config;
import com.imcode.imcms.addon.imagearchive.command.ChangeImageDataCommand;
import com.imcode.imcms.addon.imagearchive.command.SearchImageCommand;
import com.imcode.imcms.addon.imagearchive.entity.Category;
import com.imcode.imcms.addon.imagearchive.entity.Exif;
import com.imcode.imcms.addon.imagearchive.entity.Image;
import com.imcode.imcms.addon.imagearchive.entity.Keyword;
import com.imcode.imcms.addon.imagearchive.json.UploadResponse;
import com.imcode.imcms.addon.imagearchive.repository.CategoryRepository;
import com.imcode.imcms.addon.imagearchive.repository.ExifRepository;
import com.imcode.imcms.addon.imagearchive.repository.ImageRepository;
import com.imcode.imcms.addon.imagearchive.repository.KeywordsRepository;
import com.imcode.imcms.addon.imagearchive.service.Facade;
import com.imcode.imcms.addon.imagearchive.service.ImageService;
import com.imcode.imcms.addon.imagearchive.service.file.FileServiceImpl;
import com.imcode.imcms.addon.imagearchive.util.Pagination;
import com.imcode.imcms.addon.imagearchive.util.exif.ExifData;
import com.imcode.imcms.addon.imagearchive.util.exif.ExifUtils;
import com.imcode.imcms.addon.imagearchive.util.exif.Flash;
import com.imcode.imcms.addon.imagearchive.util.image.ImageInfo;
import com.imcode.imcms.addon.imagearchive.util.image.ImageOp;
import com.imcode.imcms.addon.imagearchive.validator.ChangeImageDataValidator;
import com.imcode.imcms.addon.imagearchive.validator.ImageUploadValidator;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Session;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.mock.web.MockMultipartFile;
import org.springframework.orm.jpa.JpaTransactionManager;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.validation.ValidationUtils;

import javax.persistence.EntityManager;
import javax.persistence.criteria.*;
import java.io.*;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

@Service
@Transactional
public class ImageServiceImpl implements ImageService {
    private static final Log log = LogFactory.getLog(ImageServiceImpl.class);

    private static final Pattern LIKE_SPECIAL_PATTERN = Pattern.compile("([%_|])");

    @Autowired
    JpaTransactionManager transactionManager;

    @Autowired
    private Facade facade;
    @Autowired
    private ImageRepository imageRepository;
    @Autowired
    private CategoryRepository categoryRepository;
    @Autowired
    private KeywordsRepository keywordsRepository;
    @Autowired
    private ExifRepository exifRepository;

    //    @Autowired
//    private PlatformTransactionManager txManager;
    @Autowired
    private Config config;


    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public Image findById(final Long imageId) {
        return imageRepository.findOne(imageId);
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public Image findById(final Long imageId, final List<Long> categoryIds) {
        Image image = findById(imageId);

        if (image != null) {
            List<Category> categories = image.getCategories();
            //if imege do not have categories, then show it anyway
            if (categories == null || categories.size() <= 0) {
                return image;
            }

            for (Category category : categories) {
                if (categoryIds.contains(category.getId())) {
                    return image;
                }
            }
        }

        return null;
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public Exif findExifByPK(Long imageId) {
        Image image = imageRepository.findByIDAndFetchExif(imageId);

        if (image != null) {
            return image.getExif();
        }

        return null;
//        Exif exif = exifRepository.findOne(imageId);
//        return exif;
    }

    public Image createImage(File tempFile, ImageInfo imageInfo, String imageName, short status) {
        Image image = create(tempFile, imageInfo, imageName);
        image.setStatus(status);

        image = imageRepository.save(image);

        if (!facade.getFileService().storeImage(tempFile, image.getId(), false)) {
            return null;
        }

        return image;
    }

    private Image create(File tempFile, ImageInfo imageInfo, String imageName) {
        Image image = new Image();

        String copyright = "";
        String description = "";
        String artist = "";
        String manufacturer = null;
        String model = null;
        String compression = null;
        Double exposure = null;
        String exposureProgram = null;
        Float fStop = null;
        Date dateOriginal = null;
        Date dateDigitized = null;
        Flash flash = null;
        Float focalLength = null;
        String colorSpace = null;
        Integer xResolution = null;
        Integer yResolution = null;
        Integer resolutionUnit = null;
        Integer pixelXDimension = null;
        Integer pixelYDimension = null;
        Integer ISO = null;

        ExifData data = ExifUtils.getExifData(tempFile);
        if (data != null) {
            copyright = StringUtils.substring(data.getCopyright(), 0, 255);
            description = StringUtils.substring(data.getDescription(), 0, 255);
            artist = StringUtils.substring(data.getArtist(), 0, 255);
            xResolution = data.getxResolution();
            yResolution = data.getyResolution();
            manufacturer = data.getManufacturer();
            model = data.getModel();
            compression = data.getCompression();
            exposure = data.getExposure();
            exposureProgram = data.getExposureProgram();
            fStop = data.getfStop();
            flash = data.getFlash();
            focalLength = data.getFocalLength();
            colorSpace = data.getColorSpace();
            resolutionUnit = data.getResolutionUnit();
            pixelXDimension = data.getPixelXDimension();
            pixelYDimension = data.getPixelYDimension();
            dateOriginal = data.getDateOriginal();
            dateDigitized = data.getDateDigitized();
            ISO = data.getISO();
        }

//        Exif changedExif = new Exif(xResolution, yResolution, description, artist, copyright,
//                    manufacturer, model, compression, exposure, exposureProgram, fStop, flash, focalLength, colorSpace,
//                    resolutionUnit, pixelXDimension, pixelYDimension, dateOriginal, dateDigitized, ISO);
        Exif originalExif = new Exif(xResolution, yResolution, description, artist, copyright,
                manufacturer, model, compression, exposure, exposureProgram, fStop, flash, focalLength, colorSpace,
                resolutionUnit, pixelXDimension, pixelYDimension, dateOriginal, dateDigitized, ISO);

//        originalExif = exifRepository.save(originalExif);
        String uploadedBy = "";
        image.setUploadedBy(uploadedBy);
//        image.setUsersId(user.getUserId());

        image.setName(StringUtils.substring(imageName, 0, 255));
        image.setFormat(imageInfo.getFormat().getOrdinal());
        image.setFileSize((int) tempFile.length());
        image.setWidth(imageInfo.getWidth());
        image.setHeight(imageInfo.getHeight());
        image.setExif(originalExif);
//        imageRepository.save(image);
//        imageRepository.save(image);

        return image;
    }

    public List<Image> createImagesFromZip(File tempFile) {
        ZipFile zip = null;
        List<Image> createdImages = new ArrayList<Image>();

        try {
            zip = new ZipFile(tempFile, ZipFile.OPEN_READ);

            Enumeration<? extends ZipEntry> entries = zip.entries();
            while (entries.hasMoreElements()) {
                ZipEntry entry = entries.nextElement();

                String fileName = entry.getName();
                Matcher matcher = FileServiceImpl.FILENAME_PATTERN.matcher(fileName);

                if (fileName.startsWith(FileServiceImpl.OSX_RESOURCE_FORK_PREFIX) || !matcher.matches() || StringUtils.isEmpty((fileName = matcher.group(1).trim()))) {
                    continue;
                }

                String extension = StringUtils.substringAfterLast(fileName, ".").toLowerCase();
                if (!FileServiceImpl.IMAGE_EXTENSIONS_SET.contains(extension)) {
                    continue;
                }

                File entryFile = facade.getFileService().createTemporaryFile("zipEntryTmp");

                InputStream inputStream = null;
                OutputStream outputStream = null;
                try {
                    inputStream = zip.getInputStream(entry);
                    outputStream = new BufferedOutputStream(FileUtils.openOutputStream(entryFile));

                    IOUtils.copy(inputStream, outputStream);
                    outputStream.flush();
                    IOUtils.closeQuietly(outputStream);
                    IOUtils.closeQuietly(inputStream);
                    ImageInfo imageInfo = ImageOp.getImageInfo(config, entryFile);
                    if (imageInfo == null || imageInfo.getFormat() == null
                            || imageInfo.getWidth() < 1 || imageInfo.getHeight() < 1) {
                        continue;
                    }
                    Image image = this.createImage(entryFile, imageInfo, fileName, Image.STATUS_ACTIVE);
                    if (image != null) {
                        createdImages.add(image);
                    }

                } catch (Exception ex) {
                    log.warn(ex.getMessage(), ex);
                    entryFile.delete();
                } finally {
                    IOUtils.closeQuietly(outputStream);
                    IOUtils.closeQuietly(inputStream);
                    entryFile.delete();
                }
            }

        } catch (Exception ex) {
            log.warn(ex.getMessage(), ex);
        } finally {
            if (zip != null) {
                try {
                    zip.close();
                } catch (IOException ex) {
                    log.warn(ex.getMessage(), ex);
                }
            }
        }

        return createdImages;
    }

    public void createImages(List<Object[]> tuples) {
        for (Object[] tuple : tuples) {
            File tempFile = (File) tuple[0];
            ImageInfo imageInfo = (ImageInfo) tuple[1];
            String imageName = (String) tuple[2];
            createImage(tempFile, imageInfo, imageName, Image.STATUS_ACTIVE);
        }
    }

    public void deleteImage(Long imageId) {
        imageRepository.delete(imageId);

        facade.getFileService().deleteImage(imageId);
    }

    public void updateFullData(final Image image, final List<Long> categoryIds, final List<String> imageKeywords) {

        List<Category> categories = categoryRepository.findAll(categoryIds);
        List<Keyword> keywords = new LinkedList<Keyword>();

        for (String keywordString : imageKeywords) {
            Keyword keyword = null;
            if (keyword != null) {
                keyword = keywordsRepository.findByName(keywordString);
            } else {
                keyword = keywordsRepository.save(new Keyword(keywordString));
            }

            keywords.add(keyword);
        }

        image.setCategories(categories);
        image.setKeywords(keywords);

        imageRepository.save(image);
    }

    public void updateData(final Image image, final List<Long> categoryIds, final List<String> imageKeywords) {
        updateFullData(image, categoryIds, imageKeywords);
    }

    private void updateImageCategories(Image image, List<Long> categoryIds) {
        List<Category> categories = categoryRepository.findAll(categoryIds);
        image.setCategories(categories);
        imageRepository.save(image);
    }

    private void updateImageKeywords(Session session, Image image, List<String> imageKeywords) {
        List<Keyword> keywords = new LinkedList<Keyword>();

        for (String keywordString : imageKeywords) {
            Keyword keyword = null;
            if (keyword != null) {
                keyword = keywordsRepository.findByName(keywordString);
            } else {
                keyword = keywordsRepository.save(new Keyword(keywordString));
            }

            keywords.add(keyword);
        }

        image.setKeywords(keywords);

        imageRepository.save(image);
    }

    public void archiveImage(final Long imageId) {
        setStatus(imageId, Image.STATUS_ARCHIVED);
    }

    private void setStatus(Long imageId, short status) {
        Image image = imageRepository.findOne(imageId);
        if (image != null) {
            image.setStatus(status);
            imageRepository.save(image);
        }
    }

    public void unarchiveImage(final Long imageId) {
        setStatus(imageId, Image.STATUS_ACTIVE);
    }

    public List<Image> getAllImages() {
        return imageRepository.findAll();
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public Long searchImagesCount(final List<Long> categoryIds) {
        Set<Image> imageSet = new HashSet<Image>();

        List<Category> categories = categoryRepository.findAll(categoryIds);
        for (Category category : categories) {
            imageSet.addAll(category.getImages());
        }
        return (long) imageSet.size();

    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public List<Image> searchImages(final Pagination pag, SearchImageCommand command) {
        PageRequest pageRequest = new PageRequest(pag.getCurrentPage(), pag.getPageSize());
        HashSet<Image> foundImages = new HashSet<Image>();

       /* int from = pageRequest.getOffset();
        int to = foundImages.size() > pageRequest.getOffset() + pag.getPageSize() ?
                pageRequest.getOffset() + pag.getPageSize() : foundImages.size();*/

        // return new LinkedList<Image>(new LinkedList<Image>(foundImages).subList(from, to));
        EntityManager entityManager = transactionManager.getEntityManagerFactory().createEntityManager();
        CriteriaBuilder builder = entityManager.getCriteriaBuilder();

        CriteriaQuery<Image> criteriaQuery = builder.createQuery(Image.class);

        Root<Image> root = criteriaQuery.from(Image.class);
        Join<Image, Category> categoriesJoin = root.join("categories", JoinType.LEFT);
        Join<Image, Keyword> keywordsJoin = root.join("keywords", JoinType.LEFT);
        Join<Image, Exif> exifJoin = root.join("exif", JoinType.LEFT);
        Predicate whereClause;
        {
            List<Integer> categoryIds = command.getCategoryIds();
            boolean hasNoCategory = categoryIds.remove(Integer.valueOf(-2));
            whereClause = hasNoCategory ?
                    categoryIds.size() > 0 ?
                            builder.or(
                                    categoriesJoin.<Integer>get("id").in(command.getCategoryIds()),
                                    builder.isEmpty(root.<Collection>get("categories"))
                            ) : builder.isEmpty(root.<Collection>get("categories"))
                    : categoriesJoin.<Integer>get("id").in(command.getCategoryIds());
        }
        //if(command.getKe)
        //keywordsJoin.<Integer>get("id").in(command.getKeywordId())
        if (StringUtils.isNoneEmpty(command.getFreetext())) {
            String formattedFreeText = "%" + command.getFreetext().trim() + "%";
            Predicate likePredicate = builder.like(root.<String>get("name"), formattedFreeText);
            if (!command.isFileNamesOnly()) {
                likePredicate = builder.or(likePredicate,
                        builder.like(root.<String>get("altText"), formattedFreeText),
                        builder.like(exifJoin.<String>get("description"), formattedFreeText),
                        builder.like(exifJoin.<String>get("artist"), formattedFreeText),
                        builder.like(exifJoin.<String>get("copyright"), formattedFreeText),
                        builder.like(exifJoin.<String>get("manufacturer"), formattedFreeText),
                        builder.like(exifJoin.<String>get("model"), formattedFreeText),
                        builder.like(exifJoin.<String>get("compression"), formattedFreeText),
                        builder.like(exifJoin.<String>get("colorSpace"), formattedFreeText),
                        builder.like(exifJoin.<String>get("colorSpace"), formattedFreeText));
            }
            whereClause = builder.and(whereClause, likePredicate);
        }
        criteriaQuery.where(whereClause);

        /*for (Keyword keyword : keywords) {
            criteriaQuery.where(builder.isMember(keyword, rootKeywords));
        }*/
        return entityManager.createQuery(criteriaQuery.select(root).distinct(true))
                .setFirstResult(pageRequest.getOffset())
                .setMaxResults(pageRequest.getPageSize()).getResultList();
    }

    @SuppressWarnings("unchecked")
    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public List<Category> findImageCategories(final Long imageId) {
        Image image = imageRepository.findOne(imageId);
        List<Category> categories = new LinkedList<Category>();

        if (image != null) {
            return image.getCategories();
        }

        return categories;
    }

    @SuppressWarnings("unchecked")
    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public List<Category> findAvailableImageCategories(final Long imageId) {
        return findImageCategories(imageId);
    }

    @SuppressWarnings("unchecked")
    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public List<String> findAvailableKeywords(final Long imageId) {
        return findImageKeywords(imageId);
    }

    @SuppressWarnings("unchecked")
    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public List<String> findImageKeywords(final Long imageId) {
        Image image = imageRepository.findOne(imageId);
        List<String> keywordStrings = new LinkedList<String>();

        if (image != null) {
            List<Keyword> keywords = image.getKeywords();
            if (keywords != null) {
                for (Keyword keyword : image.getKeywords()) {
                    keywordStrings.add(keyword.getName());
                }
            }
        }

        return keywordStrings;
    }

    public void uploadHandler(
            byte[] file,
            String fileName,
            Integer fileCount,
            BindingResult result,
            ChangeImageDataCommand changeData,
            BindingResult dataResult) {

        MockMultipartFile multipartFile = new MockMultipartFile(fileName, file);

        ImageUploadValidator validator = new ImageUploadValidator(facade);
        ValidationUtils.invokeValidator(validator, multipartFile, result);

        if ((fileCount > 0 || validator.isZipFile()) && changeData.isSubmitted()) {
            //Валидатор информации о картинке
            ChangeImageDataValidator dataValidator = new ChangeImageDataValidator(facade);
            ValidationUtils.invokeValidator(dataValidator, changeData, dataResult);
        }

        UploadResponse uploadResponse = new UploadResponse();
        if (!result.hasErrors() && !dataResult.hasErrors()) {
            try {
                if (validator.isZipFile()) {
                    List<Image> images = facade.getImageService().createImagesFromZip(validator.getTempFile());
                    if (changeData.isSubmitted()) {
                        for (Image image : images) {
                            if (image != null) {
                                changeData.setImageNm(image.getName());
                                changeData.toImage(image);
                                facade.getImageService().updateData(image, changeData.getCategoryIds(), changeData.getImageKeywordNames());
                            }
                        }
                    }

                } else {
                    Image image;
                    if (fileCount > 1) {
                        image = facade.getImageService().createImage(validator.getTempFile(), validator.getImageInfo(), validator.getImageName(), Image.STATUS_ACTIVE);
                    } else {
                        image = facade.getImageService().createImage(validator.getTempFile(), validator.getImageInfo(), validator.getImageName(), Image.STATUS_UPLOADED);
                        if (changeData.isSubmitted()) {
                            if (image != null) {
                                changeData.toImage(image);
                                facade.getImageService().updateData(image, changeData.getCategoryIds(), changeData.getImageKeywordNames());
                            }
                        }
                    }

                    if (image == null) {
                        result.rejectValue("file", "addImage.invalidImageError");
                    } else {
                        if (changeData.isSubmitted()) {
                            changeData.setImageNm(image.getName());
                            changeData.toImage(image);
                            facade.getImageService().updateData(image, changeData.getCategoryIds(), changeData.getImageKeywordNames());
                        }
                    }
                }

            } catch (Exception ex) {
                log.fatal(ex.getMessage(), ex);
                result.rejectValue("file", "addImage.invalidImageError");
            } finally {
                validator.getTempFile().delete();
            }
        }

        if (result.hasErrors()) {
            List<String> errors = new ArrayList<String>();
            for (Object errorObj : result.getFieldErrors()) {
                FieldError error = (FieldError) errorObj;
                errors.add(facade.getCommonService().getMessage(error.getCode(), null, error.getArguments()));
            }
            uploadResponse.setImageErrors(errors);
        }

        if (dataResult.hasErrors()) {
            Map<String, String> errors = new HashMap<String, String>();
            for (Object errorObj : dataResult.getFieldErrors()) {
                FieldError error = (FieldError) errorObj;
                errors.put(error.getField(), facade.getCommonService().getMessage(error.getCode(), null, error.getArguments()));
            }
            uploadResponse.setDataErrors(errors);
        }

    }

    public File uploadFile(byte[] data) {
        File tempFile = facade.getFileService().createTemporaryFile("img_upload_" + Math.round(Math.random() * 100000));
        try {
            FileOutputStream output = new FileOutputStream(tempFile);
            IOUtils.write(data, output);
        } catch (IOException e) {
            return null;
        }
        return tempFile;
    }
}
