package com.imcode.imcms.addon.imagearchive.util;


import com.fasterxml.jackson.databind.ObjectMapper;
import com.imcode.imcms.addon.imagearchive.Config;
import com.imcode.imcms.addon.imagearchive.entity.*;
import com.imcode.imcms.addon.imagearchive.service.Facade;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.math.BigInteger;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class Utils {
    private static final Log log = LogFactory.getLog(Utils.class);

    private static final Pattern DOMAIN_PATTERN = Pattern.compile("^.*?([^.]+?\\.[^.]+)$");
    private static final Pattern IP_PATTERN = Pattern.compile("^\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}\\.\\d{1,3}$");
    private static final String DEFAULT_ENCODING = "UTF-8";


    public static void addNoCacheHeaders(HttpServletResponse response) {
        response.setHeader("Cache-Control", "no-cache, must-revalidate, max_age=0, no-store");
        response.setHeader("Pragma", "no-cache");
        response.setHeader("Expires", "0");
    }

    public static void sendErrorCode(HttpServletResponse response, int statusCode) {
        try {
            response.sendError(statusCode);
        } catch (IOException ex) {
            log.warn(ex.getMessage(), ex);
        }
    }
    
    public static String makeKey(Class<?> klass, String suffix) {
        return String.format("%s.%s", klass.getName(), suffix);
    }
    
    public static Date min(Date date1, Date date2) {
        if (date1 == null && date2 == null) {
            return null;
        }
        
        if (date1 == null) {
            return date2;
        } else if (date2 == null) {
            return date1;
        }
        
        return (date1.getTime() < date2.getTime() ? date1 : date2);
    }
    
    public static Date max(Date date1, Date date2) {
        if (date1 == null && date2 == null) {
            return null;
        }
        
        if (date1 == null) {
            return date2;
        } else if (date2 == null) {
            return date1;
        }
        
        return (date1.getTime() < date2.getTime() ? date2 : date1);
    }
    
    public static void redirectToLogin(HttpServletResponse response, Facade facade) {
        try {
            Config config = facade.getConfig();
            response.sendRedirect(String.format("http://%s/%s", config.getImcmsRootUrl(), config.getImcmsLoginUrlPath()));
        } catch (IOException ex) {
            log.warn(ex.getMessage(), ex);
        }
    }

    public static void setCookieDomain(HttpServletRequest request, Cookie cookie) {
        String serverName = request.getServerName();
        if (!IP_PATTERN.matcher(serverName).matches()) {
            Matcher matcher = DOMAIN_PATTERN.matcher(serverName);

            if (matcher.matches()) {
                cookie.setDomain("." + matcher.group(1));
            }
        }
    }

    public static String encodeUrl(String value) {
        try {
            return URLEncoder.encode(value, "UTF-8");
        } catch (UnsupportedEncodingException ex) {
            log.warn(ex.getMessage(), ex);
        }
        
        return null;
    }
    
    public static String decodeUrl(String value) {
        try {
            return URLDecoder.decode(value, "UTF-8");
        } catch (UnsupportedEncodingException ex) {
            log.warn(ex.getMessage(), ex);
        }
        
        return null;
    }

    public static List<Long> getLibrarySubLibriesIds(Library rootLibrary, List<Library> allLibraries) {
        List<Long> subLibrariesIds = new ArrayList<Long>();

        String path = rootLibrary.getFilepath();
        if (path != null) {
            File file = new File(path, rootLibrary.getFolderNm());
            try {
                getSubdirs(file, new FileFilter() {
                    public boolean accept(File file) {
                        String name = file.getName();

                        return file.isDirectory() && name.length() <= 255;
                    }
                }, subLibrariesIds, allLibraries);
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

        return subLibrariesIds;
    }

    private static void getSubdirs(File file, FileFilter filter, List<Long> subLibIds, List<Library> allLibs) throws IOException {
        if (file == null) {
            return;
        }

        File[] subDirsTmp = file.listFiles(filter);
        if (subDirsTmp == null) {
            subDirsTmp = new File[0];
        }

        List<File> subdirs = Arrays.asList(subDirsTmp);
        subdirs = new ArrayList<File>(subdirs);

        for (File subdir : subdirs) {
            Library subLib = matchPathToLibrary(subdir, allLibs);
            if(subLib == null) {
                return;
            }
            subLibIds.add(subLib.getId());
            getSubdirs(subdir, filter, subLibIds, allLibs);
        }
    }

    private static Library matchPathToLibrary(File path, List<Library> allLibraries) {
        for(Library lib: allLibraries) {
            if(lib.getFilepath() != null) {
                File f = new File(lib.getFilepath(), lib.getFolderNm());
                if(path.equals(f)) {
                    return lib;
                }
            }
        }

        return null;
    }

    public static void writeJSON(Object object, HttpServletResponse response) {
        response.setContentType("application/json;charset=UTF-8");
        ObjectMapper mapper = new ObjectMapper();
        try {
            mapper.writeValue(response.getWriter(), object);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public static List<Image> isInArchive(File img, Facade facade) {
        List<Image> sameImages = new ArrayList<Image>();
        List<Image> archiveImages = facade.getImageService().getAllImages();
        for (Image image : archiveImages) {
            if (img != null && image.getName().equals(img.getName()) && image.getFileSize() == img.length()) {
                sameImages.add(image);
            }
        }

        return sameImages;
    }

    public static List<Category> getCategoriesRequiredToUse(Image img, Facade facader) {
//        if(user.isSuperadmin() || Utils.isImageAdmin(user, facade)) {
//            return Collections.emptyList();
//        }

//        List<Category> userCategories = facade.getRoleService().findCategories(user, Roles.ALL_PERMISSIONS);
        List<Category> userCategories = facader.getCategoryService().getCategories();
        List<Category> imageCategories = img.getCategories();
        List<Category> categoriesesUserCantUse = Collections.emptyList();
        if (imageCategories != null) {
            categoriesesUserCantUse = (List<Category>) CollectionUtils.subtract(imageCategories, userCategories);
        }

        /* Not an image uploaded by the user, return image's categories the user can't use or edit(so they know what to
         get to be able to activate) */
//        if (img.getUsersId() != user.getUserId() && categoriesesUserCantUse.size() > 0) {
//            return categoriesesUserCantUse;
//        }

        return categoriesesUserCantUse;
    }

    public static boolean canAccessPreferences(Facade facade) {
//        return user.isSuperadmin() || isImageAdmin(user, facade);
        return true;
    }

    public static boolean isImageAdmin(Facade facade) {
//        Roles role = facade.getRoleService().getRoleByName("Bildadmin");
//        return role != null && user.hasRole(role);
        return true;
    }

    private Utils() {
    }

    public static String md5Custom(String st) {
        MessageDigest messageDigest = null;
        byte[] digest = new byte[0];

        try {
            messageDigest = MessageDigest.getInstance("MD5");
            messageDigest.reset();
            messageDigest.update(st.getBytes());
            digest = messageDigest.digest();
        } catch (NoSuchAlgorithmException e) {
            // тут можно обработать ошибку
            // возникает она если в передаваемый алгоритм в getInstance(,,,) не существует
            e.printStackTrace();
        }

        BigInteger bigInt = new BigInteger(1, digest);
        String md5Hex = bigInt.toString(16);

        while( md5Hex.length() < 32 ){
            md5Hex = "0" + md5Hex;
        }

        return md5Hex;
    }



    public static String buildQueryString(final Map<String, Object> map) {
        try {
            final Iterator<Map.Entry<String, Object>> it = map.entrySet().iterator();
            final StringBuilder sb = new StringBuilder(map.size() * 8);
            while (it.hasNext()) {
                final Map.Entry<String, Object> entry = it.next();
                final String key = entry.getKey();
                if (key != null) {
                    sb.append(URLEncoder.encode(key, DEFAULT_ENCODING));
                    sb.append("=");
                    final Object value = entry.getValue();
                    final String valueAsString = value != null ? URLEncoder.encode(value.toString(), DEFAULT_ENCODING) : "";
                    sb.append(valueAsString);
                    if (it.hasNext()) {
                        sb.append("&");
                    }
                } else {
                    // Do what you want...for example:
                    assert false : String.format("Null key in query map: %s", map.entrySet());
                }
            }
            return sb.toString();
        } catch (final UnsupportedEncodingException e) {
            throw new UnsupportedOperationException(e);
        }
    }

    public static String buildQueryStringMulti(final Map<String, String[]> map) {
        try {
            final StringBuilder sb = new StringBuilder(map.size() * 8);
            for (final Iterator<Map.Entry<String, String[]>> mapIterator = map.entrySet().iterator(); mapIterator.hasNext();) {
                final Map.Entry<String, String[]> entry = mapIterator.next();
                final String key = entry.getKey();
                if (key != null) {
                    final String keyEncoded = URLEncoder.encode(key, DEFAULT_ENCODING);
                    final List<String> values = Arrays.asList(entry.getValue());
                    sb.append(keyEncoded);
                    sb.append("=");
                    if (values != null) {
                        for (final Iterator<String> listIt = values.iterator(); listIt.hasNext();) {
                            final Object valueObject = listIt.next();
                            sb.append(valueObject != null ? URLEncoder.encode(valueObject.toString(), DEFAULT_ENCODING) : "");
                            if (listIt.hasNext()) {
                                sb.append("&");
                                sb.append(keyEncoded);
                                sb.append("=");
                            }
                        }
                    }
                    if (mapIterator.hasNext()) {
                        sb.append("&");
                    }
                } else {
                    // Do what you want...for example:
                    assert false : String.format("Null key in query map: %s", map.entrySet());
                }
            }
            return sb.toString();
        } catch (final UnsupportedEncodingException e) {
            throw new UnsupportedOperationException(e);
        }
    }

    public static <T> List<T> asList(T... a) {

        if (a == null) {
            return new ArrayList<T>();
        } else {
            return Arrays.asList(a);
        }

    }

    public static Object doReturn(boolean json, ModelAndView mav) {
        if (json)
            return new FakeModelAndView2(mav);
        else
            return mav;
    }

}
