package com.imcode.imcms.addon.imagearchive.service.jpa;

import com.imcode.imcms.addon.imagearchive.Config;
import com.imcode.imcms.addon.imagearchive.command.ChangeImageDataCommand;
import com.imcode.imcms.addon.imagearchive.command.SearchImageCommand;
import com.imcode.imcms.addon.imagearchive.entity.Category;
import com.imcode.imcms.addon.imagearchive.entity.Exif;
import com.imcode.imcms.addon.imagearchive.entity.Image;
import com.imcode.imcms.addon.imagearchive.entity.Keyword;
import com.imcode.imcms.addon.imagearchive.json.UploadResponse;
import com.imcode.imcms.addon.imagearchive.repository.CategoryRepository;
import com.imcode.imcms.addon.imagearchive.repository.ExifRepository;
import com.imcode.imcms.addon.imagearchive.repository.ImageRepository;
import com.imcode.imcms.addon.imagearchive.repository.KeywordsRepository;
import com.imcode.imcms.addon.imagearchive.service.Facade;
import com.imcode.imcms.addon.imagearchive.service.ImageService;
import com.imcode.imcms.addon.imagearchive.service.file.FileServiceImpl;
import com.imcode.imcms.addon.imagearchive.util.Pagination;
import com.imcode.imcms.addon.imagearchive.util.exif.ExifData;
import com.imcode.imcms.addon.imagearchive.util.exif.ExifUtils;
import com.imcode.imcms.addon.imagearchive.util.exif.Flash;
import com.imcode.imcms.addon.imagearchive.util.image.ImageInfo;
import com.imcode.imcms.addon.imagearchive.util.image.ImageOp;
import com.imcode.imcms.addon.imagearchive.validator.ChangeImageDataValidator;
import com.imcode.imcms.addon.imagearchive.validator.ImageUploadValidator;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Session;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.mock.web.MockMultipartFile;
import org.springframework.orm.jpa.JpaTransactionManager;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.validation.ValidationUtils;

import javax.persistence.EntityManager;
import javax.persistence.criteria.*;
import java.io.*;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

@Service
@Transactional
public class ImageServiceImpl implements ImageService {
    private static final Log log = LogFactory.getLog(ImageServiceImpl.class);

    private static final Pattern LIKE_SPECIAL_PATTERN = Pattern.compile("([%_|])");

    @Autowired
    JpaTransactionManager transactionManager;

    @Autowired
    private Facade facade;
    @Autowired
    private ImageRepository imageRepository;
    @Autowired
    private CategoryRepository categoryRepository;
    @Autowired
    private KeywordsRepository keywordsRepository;
    @Autowired
    private ExifRepository exifRepository;

    //    @Autowired
//    private PlatformTransactionManager txManager;
    @Autowired
    private Config config;


    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public Image findById(final Long imageId) {
        return imageRepository.findOne(imageId);
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public Image findById(final Long imageId, final List<Long> categoryIds) {
        Image image = findById(imageId);

        if (image != null) {
            List<Category> categories = image.getCategories();
            //if imege do not have categories, then show it anyway
            if (categories == null || categories.size() <= 0) {
                return image;
            }

            for (Category category : categories) {
                if (categoryIds.contains(category.getId())) {
                    return image;
                }
            }
        }

        return null;
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public Exif findExifByPK(Long imageId) {
        Image image = imageRepository.findByIDAndFetchExif(imageId);

        if (image != null) {
            return image.getExif();
        }

        return null;
//        Exif exif = exifRepository.findOne(imageId);
//        return exif;
    }

    public Image createImage(File tempFile, ImageInfo imageInfo, String imageName, short status) {
        Image image = create(tempFile, imageInfo, imageName);
        image.setStatus(status);

        image = imageRepository.save(image);

        if (!facade.getFileService().storeImage(tempFile, image.getId(), false)) {
            return null;
        }

        return image;
    }

    private Image create(File tempFile, ImageInfo imageInfo, String imageName) {
        Image image = new Image();

        String copyright = "";
        String description = "";
        String artist = "";
        String manufacturer = null;
        String model = null;
        String compression = null;
        Double exposure = null;
        String exposureProgram = null;
        Float fStop = null;
        Date dateOriginal = null;
        Date dateDigitized = null;
        Flash flash = null;
        Float focalLength = null;
        String colorSpace = null;
        Integer xResolution = null;
        Integer yResolution = null;
        Integer resolutionUnit = null;
        Integer pixelXDimension = null;
        Integer pixelYDimension = null;
        Integer ISO = null;

        ExifData data = ExifUtils.getExifData(tempFile);
        if (data != null) {
            copyright = StringUtils.substring(data.getCopyright(), 0, 255);
            description = StringUtils.substring(data.getDescription(), 0, 255);
            artist = StringUtils.substring(data.getArtist(), 0, 255);
            xResolution = data.getxResolution();
            yResolution = data.getyResolution();
            manufacturer = data.getManufacturer();
            model = data.getModel();
            compression = data.getCompression();
            exposure = data.getExposure();
            exposureProgram = data.getExposureProgram();
            fStop = data.getfStop();
            flash = data.getFlash();
            focalLength = data.getFocalLength();
            colorSpace = data.getColorSpace();
            resolutionUnit = data.getResolutionUnit();
            pixelXDimension = data.getPixelXDimension();
            pixelYDimension = data.getPixelYDimension();
            dateOriginal = data.getDateOriginal();
            dateDigitized = data.getDateDigitized();
            ISO = data.getISO();
        }

//        Exif changedExif = new Exif(xResolution, yResolution, description, artist, copyright,
//                    manufacturer, model, compression, exposure, exposureProgram, fStop, flash, focalLength, colorSpace,
//                    resolutionUnit, pixelXDimension, pixelYDimension, dateOriginal, dateDigitized, ISO);
        Exif originalExif = new Exif(xResolution, yResolution, description, artist, copyright,
                manufacturer, model, compression, exposure, exposureProgram, fStop, flash, focalLength, colorSpace,
                resolutionUnit, pixelXDimension, pixelYDimension, dateOriginal, dateDigitized, ISO);

//        originalExif = exifRepository.save(originalExif);
        String uploadedBy = "";
        image.setUploadedBy(uploadedBy);
//        image.setUsersId(user.getUserId());

        image.setName(StringUtils.substring(imageName, 0, 255));
        image.setFormat(imageInfo.getFormat().getOrdinal());
        image.setFileSize((int) tempFile.length());
        image.setWidth(imageInfo.getWidth());
        image.setHeight(imageInfo.getHeight());
        image.setExif(originalExif);
//        imageRepository.save(image);
//        imageRepository.save(image);

        return image;
    }

    public List<Image> createImagesFromZip(File tempFile) {
        ZipFile zip = null;
        List<Image> createdImages = new ArrayList<Image>();

        try {
            zip = new ZipFile(tempFile, ZipFile.OPEN_READ);

            Enumeration<? extends ZipEntry> entries = zip.entries();
            while (entries.hasMoreElements()) {
                ZipEntry entry = entries.nextElement();

                String fileName = entry.getName();
                Matcher matcher = FileServiceImpl.FILENAME_PATTERN.matcher(fileName);

                if (fileName.startsWith(FileServiceImpl.OSX_RESOURCE_FORK_PREFIX) || !matcher.matches() || StringUtils.isEmpty((fileName = matcher.group(1).trim()))) {
                    continue;
                }

                String extension = StringUtils.substringAfterLast(fileName, ".").toLowerCase();
                if (!FileServiceImpl.IMAGE_EXTENSIONS_SET.contains(extension)) {
                    continue;
                }

                File entryFile = facade.getFileService().createTemporaryFile("zipEntryTmp");

                InputStream inputStream = null;
                OutputStream outputStream = null;
                try {
                    inputStream = zip.getInputStream(entry);
                    outputStream = new BufferedOutputStream(FileUtils.openOutputStream(entryFile));

                    IOUtils.copy(inputStream, outputStream);
                    outputStream.flush();
                    IOUtils.closeQuietly(outputStream);
                    IOUtils.closeQuietly(inputStream);
                    ImageInfo imageInfo = ImageOp.getImageInfo(config, entryFile);
                    if (imageInfo == null || imageInfo.getFormat() == null
                            || imageInfo.getWidth() < 1 || imageInfo.getHeight() < 1) {
                        continue;
                    }
                    Image image = this.createImage(entryFile, imageInfo, fileName, Image.STATUS_ACTIVE);
                    if (image != null) {
                        createdImages.add(image);
                    }

                } catch (Exception ex) {
                    log.warn(ex.getMessage(), ex);
                    entryFile.delete();
                } finally {
                    IOUtils.closeQuietly(outputStream);
                    IOUtils.closeQuietly(inputStream);
                    entryFile.delete();
                }
            }

        } catch (Exception ex) {
            log.warn(ex.getMessage(), ex);
        } finally {
            if (zip != null) {
                try {
                    zip.close();
                } catch (IOException ex) {
                    log.warn(ex.getMessage(), ex);
                }
            }
        }

        return createdImages;
    }

    public void createImages(List<Object[]> tuples) {
        for (Object[] tuple : tuples) {
            File tempFile = (File) tuple[0];
            ImageInfo imageInfo = (ImageInfo) tuple[1];
            String imageName = (String) tuple[2];
            createImage(tempFile, imageInfo, imageName, Image.STATUS_ACTIVE);
        }
    }

    public void deleteImage(Long imageId) {
        imageRepository.delete(imageId);

        facade.getFileService().deleteImage(imageId);
    }

    public void updateFullData(final Image image, final List<Long> categoryIds, final List<String> imageKeywords) {

        List<Category> categories = categoryRepository.findAll(categoryIds);
        List<Keyword> keywords = new LinkedList<Keyword>();

        for (String keywordString : imageKeywords) {
            Keyword keyword = null;
            if (keyword != null) {
                keyword = keywordsRepository.findByName(keywordString);
            } else {
                keyword = keywordsRepository.save(new Keyword(keywordString));
            }

            keywords.add(keyword);
        }

        image.setCategories(categories);
        image.setKeywords(keywords);

        imageRepository.save(image);


//        template.execute(new HibernateCallback() {
//            public Object doInHibernate(Session session) throws HibernateException, SQLException {
//                session.getNamedQuery("updateFullImageData")
//                        .setString("imageNm", image.getName())
//                        .setInteger("width", image.getWidth())
//                        .setInteger("height", image.getHeight())
//                        .setInteger("fileSize", image.getFileSize())
//                        .setShort("format", image.getFormat())
//                        .setString("uploadedBy", image.getUploadedBy())
//                        .setDate("licenseDt", image.getLicenseDt())
//                        .setDate("licenseEndDt", image.getLicenseEndDt())
//                        .setShort("statusActive", Image.STATUS_ACTIVE)
//                        .setString("altText", image.getAltText())
//                        .setLong("id", image.getId())
//                        .executeUpdate();
//
//                image.setStatus(Image.STATUS_ACTIVE);
//
//                Exif changedExif = image.getChangedExif();
//                Query query = session.getNamedQuery("updateImageExifFull");
//                query.setString("artist", changedExif.getArtist());
//                query.setString("description", changedExif.getDescription());
//                query.setString("copyright", changedExif.getCopyright());
//                query.setParameter("xResolution", changedExif.getxResolution(), new IntegerType());
//                query.setParameter("yResolution", changedExif.getyResolution(), new IntegerType());
//                query.setParameter("manufacturer", changedExif.getManufacturer(), new StringType());
//                query.setParameter("model", changedExif.getModel(), new StringType());
//                query.setParameter("compression", changedExif.getCompression(), new StringType());
//                query.setParameter("exposure", changedExif.getExposure(), new DoubleType());
//                query.setParameter("exposureProgram", changedExif.getExposureProgram(), new StringType());
//                query.setParameter("fStop", changedExif.getfStop(), new FloatType());
//                if(changedExif.getFlash() != null) {
//                    query.setParameter("flash", changedExif.getFlash());
//                } else {
//                    query.setParameter("flash", null, new IntegerType());
//                }
//                query.setParameter("focalLength", changedExif.getFocalLength(), new FloatType());
//                query.setParameter("colorSpace", changedExif.getColorSpace(), new StringType());
//                query.setParameter("resolutionUnit", changedExif.getResolutionUnit(), new IntegerType());
//                query.setParameter("pixelXDimension", changedExif.getPixelXDimension(), new IntegerType());
//                query.setParameter("pixelYDimension", changedExif.getPixelYDimension(), new IntegerType());
//                query.setParameter("dateOriginal", changedExif.getDateOriginal(), new TimestampType());
//                query.setParameter("dateDigitized", changedExif.getDateDigitized(), new TimestampType());
//                query.setParameter("ISO", changedExif.getISO(), new IntegerType());
//                query.setLong("imageId", image.getId());
//                query.setShort("exifType", Exif.TYPE_CHANGED);
//                query.executeUpdate();
//
//                Exif originalExif = image.getOriginalExif();
//                query = session.getNamedQuery("updateImageExifFull");
//                query.setString("artist", originalExif.getArtist());
//                query.setString("description", originalExif.getDescription());
//                query.setString("copyright", originalExif.getCopyright());
//                query.setParameter("xResolution", originalExif.getxResolution(), new IntegerType());
//                query.setParameter("yResolution", originalExif.getyResolution(), new IntegerType());
//                query.setParameter("manufacturer", originalExif.getManufacturer(), new StringType());
//                query.setParameter("model", originalExif.getModel(), new StringType());
//                query.setParameter("compression", originalExif.getCompression(), new StringType());
//                query.setParameter("exposure", originalExif.getExposure(), new DoubleType());
//                query.setParameter("exposureProgram", originalExif.getExposureProgram(), new StringType());
//                query.setParameter("fStop", originalExif.getfStop(), new FloatType());
//                if(originalExif.getFlash() != null) {
//                    query.setParameter("flash", originalExif.getFlash());
//                } else {
//                    query.setParameter("flash", null, new IntegerType());
//                }
//                query.setParameter("focalLength", originalExif.getFocalLength(), new FloatType());
//                query.setParameter("colorSpace", originalExif.getColorSpace(), new StringType());
//                query.setParameter("resolutionUnit", originalExif.getResolutionUnit(), new IntegerType());
//                query.setParameter("pixelXDimension", originalExif.getPixelXDimension(), new IntegerType());
//                query.setParameter("pixelYDimension", originalExif.getPixelYDimension(), new IntegerType());
//                query.setParameter("dateOriginal", originalExif.getDateOriginal(), new TimestampType());
//                query.setParameter("dateDigitized", originalExif.getDateDigitized(), new TimestampType());
//                query.setParameter("ISO", originalExif.getISO(), new IntegerType());
//                query.setLong("imageId", image.getId());
//                query.setShort("exifType", Exif.TYPE_ORIGINAL);
//                query.executeUpdate();
//
//                updateImageCategories(session, image, categoryIds);
//                updateImageKeywords(session, image, imageKeywords);
//
//                return null;
//            }
//        });
    }

    public void updateData(final Image image, final List<Long> categoryIds, final List<String> imageKeywords) {
        updateFullData(image, categoryIds, imageKeywords);
//        template.execute(new HibernateCallback() {
//            public Object doInHibernate(Session session) throws HibernateException, SQLException {
//                session.getNamedQuery("updateImageData")
//                        .setString("imageNm", image.getName())
//                        .setString("uploadedBy", image.getUploadedBy())
//                        .setDate("licenseDt", image.getLicenseDt())
//                        .setDate("licenseEndDt", image.getLicenseEndDt())
//                        .setShort("statusActive", Image.STATUS_ACTIVE)
//                        .setString("altText", image.getAltText())
//                        .setLong("id", image.getId())
//                        .executeUpdate();
//
//                image.setStatus(Image.STATUS_ACTIVE);
//
//                Exif exif = image.getChangedExif();
//                session.getNamedQuery("updateImageExif")
//                        .setString("artist", exif.getArtist())
//                        .setString("description", exif.getDescription())
//                        .setString("copyright", exif.getCopyright())
//                        .setLong("imageId", image.getId())
//                        .setShort("changedType", Exif.TYPE_CHANGED)
//                        .executeUpdate();
//
//                updateImageCategories(session, image, categoryIds);
//                updateImageKeywords(session, image, imageKeywords);
//
//                return null;
//            }
//        });
    }

    private void updateImageCategories(Image image, List<Long> categoryIds) {
        List<Category> categories = categoryRepository.findAll(categoryIds);
        image.setCategories(categories);
        imageRepository.save(image);
//        if (categoryIds.isEmpty()) {
//            session.createQuery("DELETE FROM ImageCategories ic WHERE ic.imageId = :imageId")
//                    .setLong("imageId", image.getId())
//                    .executeUpdate();
//            return;
//        }
//
//        session.createQuery(
//                "DELETE FROM ImageCategories ic WHERE ic.imageId = :imageId AND ic.categoryId NOT IN (:categoryIds)")
//                .setLong("imageId", image.getId())
//                .setParameterList("categoryIds", categoryIds)
//                .executeUpdate();
//
//        List<Integer> existingCategoryIds = session.createQuery(
//                "SELECT ic.categoryId FROM ImageCategories ic WHERE ic.imageId = :imageId AND ic.categoryId IN (:categoryIds) ")
//                .setLong("imageId", image.getId())
//                .setParameterList("categoryIds", categoryIds)
//                .list();
//        Set<Integer> existingCategoryIdSet = new HashSet<Integer>(existingCategoryIds);
//
//        long imageId = image.getId();
//        for (int categoryId : categoryIds) {
//            if (!existingCategoryIdSet.contains(categoryId)) {
//                ImageCategories imageCategory = new ImageCategories(imageId, categoryId);
//                session.persist(imageCategory);
//            }
//        }
//        session.flush();
    }

    private void updateImageKeywords(Session session, Image image, List<String> imageKeywords) {
        List<Keyword> keywords = new LinkedList<Keyword>();

        for (String keywordString : imageKeywords) {
            Keyword keyword = null;
            if (keyword != null) {
                keyword = keywordsRepository.findByName(keywordString);
            } else {
                keyword = keywordsRepository.save(new Keyword(keywordString));
            }

            keywords.add(keyword);
        }

        image.setKeywords(keywords);

        imageRepository.save(image);
//        long imageId = image.getId();
//        if (imageKeywords.isEmpty()) {
//            session.createQuery("DELETE FROM ImageKeywords ik WHERE ik.imageId = :imageId")
//                    .setLong("imageId", imageId)
//                    .executeUpdate();
//        } else {
//            List<Keyword> existingKeywords = session.createQuery(
//                    "SELECT k.id AS id, k.keywordNm AS keywordNm FROM Keywords k " +
//                    "WHERE k.keywordNm IN (:keywords) ")
//                    .setParameterList("keywords", imageKeywords)
//                    .setResultTransformer(Transformers.aliasToBean(Keyword.class))
//                    .list();
//            Set<String> existingSet = new HashSet<String>(existingKeywords.size());
//            for (Keyword k : existingKeywords) {
//                existingSet.add(k.getName());
//            }
//
//            List<String> newKeywordNames = new ArrayList<String>();
//            for (String keyword : imageKeywords) {
//                if (!existingSet.contains(keyword)) {
//                    newKeywordNames.add(keyword);
//                }
//            }
//
//            List<Long> keywordIds = new ArrayList<Long>(newKeywordNames.size() + existingKeywords.size());
//            for (String k : newKeywordNames) {
//                Keyword keyword = new Keyword();
//                keyword.setName(k);
//                session.persist(keyword);
//
//                keywordIds.add(keyword.getId());
//            }
//            for (Keyword k : existingKeywords) {
//                keywordIds.add(k.getId());
//            }
//
//            List<Long> existingImageKeywordIds = session.createQuery(
//                    "SELECT ik.keywordId FROM ImageKeywords ik WHERE ik.imageId = :imageId")
//                    .setLong("imageId", imageId)
//                    .list();
//            List<Long> toDelete = new ArrayList<Long>();
//            for (Long id : existingImageKeywordIds) {
//                if (keywordIds.contains(id)) {
//                    keywordIds.remove(id);
//                } else {
//                    toDelete.add(id);
//                }
//            }
//
//            if (!toDelete.isEmpty()) {
//                session.createQuery("DELETE FROM ImageKeywords ik WHERE ik.imageId = :imageId AND ik.keywordId IN (:keywordIds)")
//                        .setLong("imageId", imageId)
//                        .setParameterList("keywordIds", toDelete)
//                        .executeUpdate();
//            }
//
//            for (Long id : keywordIds) {
//                ImageKeywords ik = new ImageKeywords();
//                ik.setKeywordId(id);
//                ik.setImageId(imageId);
//                session.persist(ik);
//            }
//            session.flush();
//        }
    }

    public void archiveImage(final Long imageId) {
//        template.execute(new HibernateCallback() {
//            public Object doInHibernate(Session session) throws HibernateException, SQLException {
//                session.createQuery("UPDATE Images im SET im.status = :statusArchived WHERE im.id = :id")
//                        .setShort("statusArchived", Image.STATUS_ARCHIVED)
//                        .setLong("id", imageId)
//                        .executeUpdate();
//
//                return null;
//            }
//        });
        setStatus(imageId, Image.STATUS_ARCHIVED);
    }

    private void setStatus(Long imageId, short status) {
        Image image = imageRepository.findOne(imageId);
        if (image != null) {
            image.setStatus(status);
            imageRepository.save(image);
        }
    }

    public void unarchiveImage(final Long imageId) {
        setStatus(imageId, Image.STATUS_ACTIVE);
//        template.execute(new HibernateCallback() {
//            public Object doInHibernate(Session session) throws HibernateException, SQLException {
//                session.createQuery("UPDATE Images im SET im.status = :statusArchived WHERE im.id = :id")
//                        .setShort("statusArchived", Image.STATUS_ACTIVE)
//                        .setLong("id", imageId)
//                        .executeUpdate();
//
//                return null;
//            }
//        });

    }

    public List<Image> getAllImages() {
//        List<Image> images = (List<Image>) template.execute(new HibernateCallback() {
//            public Object doInHibernate(Session session) throws HibernateException, SQLException {
//                Query query = session.createQuery("FROM Images im WHERE im.status <> :statusUploaded")
//                        .setShort("statusUploaded", Image.STATUS_UPLOADED);
//
//                return query.list();
//            }
//        });

        return imageRepository.findAll();
    }

//    private Query buildSearchImagesQuery(Session session, SearchImageCommand command, boolean count, List<Integer> categoryIds, Users user) {
//        StringBuilder builder = new StringBuilder();
//        boolean admin = user != null && (user.isSuperadmin() || Utils.isImageAdmin(user, facade));
//
//        builder.append("SELECT ");
//
//        if (count) {
//            builder.append("count(DISTINCT im.id) ");
//        } else {
//            builder.append(
//                "DISTINCT im.id AS id, im.imageNm AS imageNm, im.width AS width, im.height AS height, " +
//                "e.artist AS artist, im.createdDt as createdDt, e.description, im.fileSize AS fileSize ");
//        }
//
//        builder.append("FROM Images im ");
//
//        List<Integer> categoryId = command.getCategoryIds();
//        if (admin) {
//            if (SearchImageCommand.CATEGORY_NO_CATEGORY.equals(categoryId)) {
//                builder.append("LEFT OUTER JOIN im.categories c ");
//            } else if (!SearchImageCommand.CATEGORY_ALL.equals(categoryId)) {
//                builder.append("INNER JOIN im.categories c ");
//            } else {
//                builder.append("LEFT OUTER JOIN im.categories c ");
//            }
//        } else if (SearchImageCommand.CATEGORY_NO_CATEGORY.equals(categoryId) || SearchImageCommand.CATEGORY_ALL.equals(categoryId)) {
//            if (user == null) {
//                builder.append("INNER JOIN im.categories c ");
//            } else {
//                builder.append("LEFT OUTER JOIN im.categories c ");
//            }
//        } else {
//            builder.append("INNER JOIN im.categories c ");
//        }
//
//        long keywordId = command.getKeywordId();
//        if (keywordId != SearchImageCommand.KEYWORD_ALL) {
//            builder.append("INNER JOIN im.keywords k ");
//        } else {
//            builder.append("LEFT OUTER JOIN im.keywords k ");
//        }
//
//        builder.append(", Exif e WHERE e.imageId = im.id AND e.type = :changedType ");
//
//        String artist = command.getArtist();
//        if (artist != null) {
//            builder.append("AND lower(e.artist) = lower(:artist) ");
//        }
//
//        Short status;
//        switch (command.getShow()) {
//            case SearchImageCommand.SHOW_ERASED:
//                builder.append("AND im.status = :status ");
//                status = Image.STATUS_ARCHIVED;
//                break;
//            case SearchImageCommand.SHOW_NEW:
//                builder.append("AND im.createdDt >= current_date() ");
//            case SearchImageCommand.SHOW_WITH_VALID_LICENCE:
//                builder.append("AND (im.licenseEndDt <= current_date() OR im.licenseEndDt IS NULL OR im.licenseEndDt = '') ");
//            default:
//                builder.append("AND im.status = :status ");
//                status = Image.STATUS_ACTIVE;
//        }
//
//
//
//        if (admin) {
//            if (SearchImageCommand.CATEGORY_NO_CATEGORY.equals(categoryId)) {
//                builder.append("AND im.categories IS EMPTY ");
//            } else if (!SearchImageCommand.CATEGORY_ALL.equals(categoryId)) {
//                builder.append("AND c.id IN (:categoryId) ");
//            }
//        } else if (SearchImageCommand.CATEGORY_ALL.equals(categoryId)) {
//            builder.append("AND (");
//
//            if (!categoryIds.isEmpty()) {
//                builder.append("c.id IN (:categoryIds) ");
//            }
//
//            if (!categoryIds.isEmpty() && user != null) {
//                builder.append("OR ");
//            }
//
//            if (user != null) {
//                builder.append("im.usersId = :usersId ");
//            }
//
//            builder.append(") ");
//        } else if (SearchImageCommand.CATEGORY_NO_CATEGORY.equals(categoryId)) {
//            builder.append("AND im.categories IS EMPTY AND im.usersId = :usersId ");
//        } else {
//            builder.append("AND c.id IN (:categoryId) ");
//        }
//
//        if (keywordId != SearchImageCommand.KEYWORD_ALL) {
//            builder.append("AND k.id = :keywordId ");
//        }
//
//        String freetext = command.getFreetext();
//        if (freetext != null) {
//            freetext = LIKE_SPECIAL_PATTERN.matcher(freetext).replaceAll("|$1");
//            if(command.isFileNamesOnly()) {
//                builder.append("AND (lower(im.imageNm) LIKE :freetext ESCAPE '|') ");
//            } else {
//                builder.append("AND (lower(im.imageNm) LIKE :freetext ESCAPE '|' " +
//                        "OR lower(e.description) LIKE :freetext ESCAPE '|' " +
//                        "OR lower(e.artist) LIKE :freetext ESCAPE '|' " +
//                        "OR e.xResolution LIKE :freetext ESCAPE '|' " +
//                        "OR e.yResolution LIKE :freetext ESCAPE '|' " +
//                        "OR e.copyright LIKE :freetext ESCAPE '|' " +
//                        "OR e.manufacturer LIKE :freetext ESCAPE '|' " +
//                        "OR e.model LIKE :freetext ESCAPE '|' " +
//                        "OR e.compression LIKE :freetext ESCAPE '|' " +
//                        "OR e.exposureProgram LIKE :freetext ESCAPE '|' " +
//                        "OR e.fStop LIKE :freetext ESCAPE '|' " +
//                        "OR e.focalLength LIKE :freetext ESCAPE '|' " +
//                        "OR e.colorSpace LIKE :freetext ESCAPE '|' " +
//                        "OR e.exposureProgram LIKE :freetext ESCAPE '|' " +
//                        "OR e.pixelXDimension LIKE :freetext ESCAPE '|' " +
//                        "OR e.pixelYDimension LIKE :freetext ESCAPE '|' " +
//                        "OR e.ISO LIKE :freetext ESCAPE '|' " +
//                        "OR lower(c.name) LIKE :freetext ESCAPE '|' " +
//                        "OR lower(k.keywordNm) LIKE :freetext ESCAPE '|') ");
//            }
//        }
//
//        Date licenseDt = command.getLicenseDate();
//        Date licenseEndDt = command.getLicenseEndDate();
//        if (licenseDt != null && licenseEndDt != null) {
//            Date min = Utils.min(licenseDt, licenseEndDt);
//            Date max = Utils.max(licenseDt, licenseEndDt);
//
//            licenseDt = min;
//            licenseEndDt = max;
//
//            builder.append("AND im.licenseDt <= :licenseDt AND im.licenseEndDt >= :licenseEndDt ");
//        } else if (licenseDt != null) {
//            builder.append("AND im.licenseDt >= :licenseDt ");
//        } else if (licenseEndDt != null) {
//            builder.append("AND im.licenseEndDt <= :licenseEndDt ");
//        }
//
//        Date activeDt = command.getActiveDate();
//        Date activeEndDt = command.getActiveEndDate();
//
//        if (!count) {
//            builder.append("ORDER BY ");
//            switch (command.getSortBy()) {
//                case SearchImageCommand.SORT_BY_ALPHABET:
//                    builder.append("im.imageNm ");
//                    break;
//                case SearchImageCommand.SORT_BY_ENTRY_DATE:
//                    builder.append("im.createdDt ");
//                    break;
//                default:
//                    builder.append("e.artist ");
//                    break;
//            }
//
//            switch(command.getSortOrder()) {
//                case SearchImageCommand.SORT_DESCENDING:
//                    builder.append("DESC ");
//                    break;
//                default:
//                    builder.append("ASC ");
//                    break;
//            }
//        }
//
//        Query query = session.createQuery(builder.toString())
//                .setShort("changedType", Exif.TYPE_CHANGED);
//
//        if (artist != null) {
//            query.setString("artist", artist);
//        }
//        if (status != null) {
//            query.setShort("status", status);
//        }
//
//        if (admin) {
//            if (!SearchImageCommand.CATEGORY_NO_CATEGORY.equals(categoryId) && !SearchImageCommand.CATEGORY_ALL.equals(categoryId)) {
//                query.setParameterList("categoryId", categoryId);
//            }
//        } else if (SearchImageCommand.CATEGORY_ALL.equals(categoryId)) {
//            if (!categoryIds.isEmpty()) {
//                query.setParameterList("categoryIds", categoryIds);
//            }
//
//            if (user != null) {
//                query.setInteger("usersId", user.getUserId());
//            }
//        } else if (SearchImageCommand.CATEGORY_NO_CATEGORY.equals(categoryId)) {
//            query.setInteger("usersId", user.getUserId());
//        } else {
//            query.setParameterList("categoryId", categoryId);
//        }
//
//        if (keywordId != SearchImageCommand.KEYWORD_ALL) {
//            query.setLong("keywordId", keywordId);
//        }
//        if (freetext != null) {
//            query.setString("freetext", "%" + freetext.toLowerCase() + "%");
//        }
//        if (licenseDt != null) {
//            query.setDate("licenseDt", licenseDt);
//        }
//        if (licenseEndDt != null) {
//            query.setDate("licenseEndDt", licenseEndDt);
//        }
//        if (activeDt != null) {
//            query.setDate("activeDt", activeDt);
//        }
//        if (activeEndDt != null) {
//            query.setDate("activeEndDt", activeEndDt);
//        }
//
//        return query;
//    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public Long searchImagesCount(final List<Long> categoryIds) {
        Set<Image> imageSet = new HashSet<Image>();

        List<Category> categories = categoryRepository.findAll(categoryIds);
        for (Category category : categories) {
            imageSet.addAll(category.getImages());
        }
        return (long) imageSet.size();

//        if (user == null && categoryIds.isEmpty()) {
//            return 0;
//        }
//
//        long count = (Long) template.execute(new HibernateCallback() {
//            public Object doInHibernate(Session session) throws HibernateException, SQLException {
//                return buildSearchImagesQuery(session, command, true, categoryIds, user)
//                        .uniqueResult();
//            }
//        });
//
//        return (int) count;
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public List<Image> searchImages(final Pagination pag, SearchImageCommand command) {
        PageRequest pageRequest = new PageRequest(pag.getCurrentPage(), pag.getPageSize());
        HashSet<Image> foundImages = new HashSet<Image>();

       /* int from = pageRequest.getOffset();
        int to = foundImages.size() > pageRequest.getOffset() + pag.getPageSize() ?
                pageRequest.getOffset() + pag.getPageSize() : foundImages.size();*/

        // return new LinkedList<Image>(new LinkedList<Image>(foundImages).subList(from, to));
        EntityManager entityManager = transactionManager.getEntityManagerFactory().createEntityManager();
        CriteriaBuilder builder = entityManager.getCriteriaBuilder();

        CriteriaQuery<Image> criteriaQuery = builder.createQuery(Image.class);

        Root<Image> root = criteriaQuery.from(Image.class);
        Join<Image, Category> categoriesJoin = root.join("categories", JoinType.LEFT);
        Join<Image, Keyword> keywordsJoin = root.join("keywords", JoinType.LEFT);
        Join<Image, Exif> exifJoin = root.join("exif", JoinType.LEFT);
        Predicate whereClause;
        {
            List<Integer> categoryIds = command.getCategoryIds();
            boolean hasNoCategory = categoryIds.remove(Integer.valueOf(-2));
            whereClause = hasNoCategory ?
                    categoryIds.size() > 0 ?
                            builder.or(
                                    categoriesJoin.<Integer>get("id").in(command.getCategoryIds()),
                                    builder.isEmpty(root.<Collection>get("categories"))
                            ) : builder.isEmpty(root.<Collection>get("categories"))
                    : categoriesJoin.<Integer>get("id").in(command.getCategoryIds());
        }
        //if(command.getKe)
        //keywordsJoin.<Integer>get("id").in(command.getKeywordId())
        if (StringUtils.isNoneEmpty(command.getFreetext())) {
            String formattedFreeText = "%" + command.getFreetext().trim() + "%";
            Predicate likePredicate = builder.like(root.<String>get("name"), formattedFreeText);
            if (!command.isFileNamesOnly()) {
                likePredicate = builder.or(likePredicate,
                        builder.like(root.<String>get("altText"), formattedFreeText),
                        builder.like(exifJoin.<String>get("description"), formattedFreeText),
                        builder.like(exifJoin.<String>get("artist"), formattedFreeText),
                        builder.like(exifJoin.<String>get("copyright"), formattedFreeText),
                        builder.like(exifJoin.<String>get("manufacturer"), formattedFreeText),
                        builder.like(exifJoin.<String>get("model"), formattedFreeText),
                        builder.like(exifJoin.<String>get("compression"), formattedFreeText),
                        builder.like(exifJoin.<String>get("colorSpace"), formattedFreeText),
                        builder.like(exifJoin.<String>get("colorSpace"), formattedFreeText));
            }
            whereClause = builder.and(whereClause, likePredicate);
        }
        criteriaQuery.where(whereClause);

        /*for (Keyword keyword : keywords) {
            criteriaQuery.where(builder.isMember(keyword, rootKeywords));
        }*/
        return entityManager.createQuery(criteriaQuery.select(root).distinct(true))
                .setFirstResult(pageRequest.getOffset())
                .setMaxResults(pageRequest.getPageSize()).getResultList();
    }

    @SuppressWarnings("unchecked")
    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public List<Category> findImageCategories(final Long imageId) {
        Image image = imageRepository.findOne(imageId);
        List<Category> categories = new LinkedList<Category>();

        if (image != null) {
            return image.getCategories();
        }

        return categories;
//        return (List<Category>) template.executeFind(new HibernateCallback() {
//            public Object doInHibernate(Session session) throws HibernateException, SQLException {
//                return session.createQuery(
//                        "SELECT c.id AS id, c.name AS name FROM ImageCategories ic INNER JOIN ic.category c " +
//                        "WHERE ic.imageId = :imageId AND c.type.imageArchive IS TRUE ORDER BY c.name ")
//                        .setLong("imageId", imageId)
//                        .setResultTransformer(Transformers.aliasToBean(Category.class))
//                        .list();
//            }
//        });
    }

//    @SuppressWarnings("unchecked")
//    @Transactional(readOnly = true)
//    public List<Category> findAvailableCategories() {
//        return (List<Category>) template.executeFind(new HibernateCallback() {
//            public Object doInHibernate(Session session) throws HibernateException, SQLException {
//                if(user != null && (user.isSuperadmin() || Utils.isImageAdmin(user, facade))) {
//                    return session.getNamedQuery("availableCategoriesForAdmin")
//                            .setResultTransformer(Transformers.aliasToBean(Category.class))
//                            .list();
//                }
//
//                Set<Integer> roleIds = facade.getUserService().getRoleIdsWithPermission(user, null, Roles.PERMISSION_CHANGE_IMAGE);
//                if (roleIds.isEmpty()) {
//                    return Collections.EMPTY_LIST;
//                }
//
//                return session.getNamedQuery("availableCategoriesForUser")
//                        .setResultTransformer(Transformers.aliasToBean(Category.class))
//                        .setParameterList("roleIds", roleIds)
//                        .list();
//            }
//        });
//    }

    @SuppressWarnings("unchecked")
    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public List<Category> findAvailableImageCategories(final Long imageId) {
        return findImageCategories(imageId);
//        return (List<Category>) template.executeFind(new HibernateCallback() {
//            public Object doInHibernate(Session session) throws HibernateException, SQLException {
//                if (user != null && (user.isSuperadmin() || Utils.isImageAdmin(user, facade))) {
//                    return session.getNamedQuery("availableImageCategoriesAdmin")
//                            .setLong("imageId", imageId)
//                            .setResultTransformer(Transformers.aliasToBean(Category.class))
//                            .list();
//                }
//
//                Set<Integer> roleIds = facade.getUserService().getRoleIdsWithPermission(user, null, Roles.PERMISSION_CHANGE_IMAGE);
//                if (roleIds.isEmpty()) {
//                    return Collections.EMPTY_LIST;
//                }
//
//                return session.getNamedQuery("availableImageCategories")
//                        .setLong("imageId", imageId)
//                        .setParameterList("roleIds", roleIds)
//                        .setResultTransformer(Transformers.aliasToBean(Category.class))
//                        .list();
//            }
//        });
    }

//    @Transactional(propagation=Propagation.SUPPORTS, readOnly=true)
//    public boolean canUseCategories(final Users user, final List<Integer> categoryIds) {
//        return (Boolean) template.execute(new HibernateCallback() {
//            public Object doInHibernate(Session session) throws HibernateException, SQLException {
//                if (user != null && (user.isSuperadmin() || Utils.isImageAdmin(user, facade))) {
//                    return true;
//                }
//
//                Set<Integer> roleIds = facade.getUserService().getRoleIdsWithPermission(user, categoryIds, Roles.PERMISSION_CHANGE_IMAGE);
//                if (roleIds.isEmpty()) {
//                    return false;
//                }
//
//                long count = (Long) session.createQuery(
//                        "SELECT count(DISTINCT cr.categoryId) FROM CategoryRoles cr " +
//                        "WHERE cr.roleId IN (:roleIds) AND cr.categoryId IN (:categoryIds) ")
//                        .setParameterList("roleIds", roleIds)
//                        .setParameterList("categoryIds", categoryIds)
//                        .uniqueResult();
//
//                return count == categoryIds.size();
//            }
//        });
//    }

    @SuppressWarnings("unchecked")
    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public List<String> findAvailableKeywords(final Long imageId) {
//        return template.executeFind(new HibernateCallback() {
//            public Object doInHibernate(Session session) throws HibernateException, SQLException {
//                return session.getNamedQuery("availableKeywords")
//                        .setLong("imageId", imageId)
//                        .list();
//            }
//        });
        return findImageKeywords(imageId);
    }

    @SuppressWarnings("unchecked")
    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public List<String> findImageKeywords(final Long imageId) {
        Image image = imageRepository.findOne(imageId);
        List<String> keywordStrings = new LinkedList<String>();

        if (image != null) {
            List<Keyword> keywords = image.getKeywords();
            if (keywords != null) {
                for (Keyword keyword : image.getKeywords()) {
                    keywordStrings.add(keyword.getName());
                }
            }
        }

        return keywordStrings;
//        return (List<String>) template.executeFind(new HibernateCallback() {
//            public Object doInHibernate(Session session) throws HibernateException, SQLException {
//                return session.createQuery(
//                        "SELECT k.keywordNm FROM ImageKeywords ik INNER JOIN ik.keyword k " +
//                                "WHERE ik.imageId = :imageId ORDER BY k.keywordNm")
//                        .setLong("imageId", imageId)
//                        .list();
//            }
//        });
    }

//    @SuppressWarnings("unchecked")
//    @Transactional(propagation=Propagation.SUPPORTS, readOnly=true)
//    public List<Keyword> findKeywords() {
//        List<Keyword> keywords = new LinkedList<Keyword>();
//
////        return (List<Keyword>) template.executeFind(new HibernateCallback() {
////            public Object doInHibernate(Session session) throws HibernateException, SQLException {
////                return session.getNamedQuery("keywordsUsedByImages")
////                        .setResultTransformer(Transformers.aliasToBean(Keyword.class))
////                        .list();
////            }
////        });
//    }


//    @Transactional(propagation=Propagation.SUPPORTS, readOnly=true)
//    public String findImageName(final long imageId) {
//        return (String) template.execute(new HibernateCallback() {
//            public Object doInHibernate(Session session) throws HibernateException, SQLException {
//                return session.createQuery("SELECT im.imageNm FROM Images im WHERE im.id = :imageId")
//                        .setLong("imageId", imageId)
//                        .uniqueResult();
//            }
//        });
//    }

//    @Transactional(propagation=Propagation.SUPPORTS, readOnly=true)
//    public String findImageAltText(final long imageId) {
//        return (String) template.execute(new HibernateCallback() {
//            public Object doInHibernate(Session session) throws HibernateException, SQLException {
//                return session.createQuery("SELECT im.altText FROM Images im WHERE im.id = :imageId")
//                        .setLong("imageId", imageId)
//                        .uniqueResult();
//            }
//        });
//    }


//    public Boolean createKeyword(final String keyword) {
//    	return (Boolean) template.execute(new HibernateCallback() {
//			public Object doInHibernate(Session session) throws HibernateException, SQLException {
//				long count = (Long) session.createQuery(
//						"SELECT COUNT(k.id) FROM Keywords k WHERE k.keywordNm = :keyword")
//						.setString("keyword", keyword)
//						.uniqueResult();
//
//				if (count == 0L) {
//					Keyword k = new Keyword();
//					k.setName(keyword);
//					session.persist(k);
//                    return true;
//				}
//
//				return false;
//			}
//		});
//    }

//    public List<Keyword> getKeywords(final int[]...ids) {
//        return (List<Keyword>) template.execute(new HibernateCallback() {
//            public Object doInHibernate(Session session) throws HibernateException, SQLException {
//                return session.createCriteria(Keyword.class, "k")
//                    .add(Restrictions.in("k.id", ids))
//                    .list();
//            }
//        });
//    }

//    public List<String> findKeywordNames(final String pattern) {
//        List<String> keywordNames = new ArrayList<String>();
//        for(Keyword keyword: findKeywords(pattern)) {
//            keywordNames.add(keyword.getName());
//        }
//
//        return keywordNames;
//    }
//
//    public List<Keyword> findKeywords(final String pattern) {
//        return (List<Keyword>) template.execute(new HibernateCallback() {
//            public Object doInHibernate(Session session) throws HibernateException, SQLException {
//                Criteria crit = session.createCriteria(Keyword.class, "k");
//
//                if(!StringUtils.isEmpty(pattern)) {
//                    crit.add(Restrictions.ilike("k.keywordNm", pattern));
//                }
//
//                crit.addOrder(Order.asc("k.keywordNm"));
//
//                return crit.list();
//            }
//        });
//    }

    public void uploadHandler(
            byte[] file,
            String fileName,
            Integer fileCount,
            BindingResult result,
            ChangeImageDataCommand changeData,
            BindingResult dataResult) {
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//        AddImageUploadCommand command = new AddImageUploadCommand();
//        command.setFileCount((Integer) uploadMap.get("fileCount"));
//        command.setFile((CommonsMultipartFile) uploadMap.get("file"));
//        command.setRedirToSearch((Boolean) uploadMap.get("redirToSearch"));
        MockMultipartFile multipartFile = new MockMultipartFile(fileName, file);

        ImageUploadValidator validator = new ImageUploadValidator(facade);
        ValidationUtils.invokeValidator(validator, multipartFile, result);

        if ((fileCount > 0 || validator.isZipFile()) && changeData.isSubmitted()) {
            //Валидатор информации о картинке
            ChangeImageDataValidator dataValidator = new ChangeImageDataValidator(facade);
            ValidationUtils.invokeValidator(dataValidator, changeData, dataResult);
        }

        UploadResponse uploadResponse = new UploadResponse();
        if (!result.hasErrors() && !dataResult.hasErrors()) {
            try {
                if (validator.isZipFile()) {
                    List<Image> images = facade.getImageService().createImagesFromZip(validator.getTempFile());
                    if (changeData.isSubmitted()) {
                        for (Image image : images) {
                            if (image != null) {
                                changeData.setImageNm(image.getName());
                                changeData.toImage(image);
                                facade.getImageService().updateData(image, changeData.getCategoryIds(), changeData.getImageKeywordNames());
                            }
                        }
                    }

                } else {
                    Image image;
                    if (fileCount > 1) {
                        image = facade.getImageService().createImage(validator.getTempFile(), validator.getImageInfo(), validator.getImageName(), Image.STATUS_ACTIVE);
                    } else {
                        image = facade.getImageService().createImage(validator.getTempFile(), validator.getImageInfo(), validator.getImageName(), Image.STATUS_UPLOADED);
                        if (changeData.isSubmitted()) {
                            if (image != null) {
                                changeData.toImage(image);
                                facade.getImageService().updateData(image, changeData.getCategoryIds(), changeData.getImageKeywordNames());
                            }
                        }
                    }

                    if (image == null) {
                        result.rejectValue("file", "addImage.invalidImageError");
                    } else {
                        if (changeData.isSubmitted()) {
                            changeData.setImageNm(image.getName());
                            changeData.toImage(image);
                            facade.getImageService().updateData(image, changeData.getCategoryIds(), changeData.getImageKeywordNames());
                        }
                    }
                }

            } catch (Exception ex) {
                log.fatal(ex.getMessage(), ex);
                result.rejectValue("file", "addImage.invalidImageError");
            } finally {
                validator.getTempFile().delete();
            }
        }

        if (result.hasErrors()) {
            List<String> errors = new ArrayList<String>();
            for (Object errorObj : result.getFieldErrors()) {
                FieldError error = (FieldError) errorObj;
                errors.add(facade.getCommonService().getMessage(error.getCode(), null, error.getArguments()));
            }
            uploadResponse.setImageErrors(errors);
        }

        if (dataResult.hasErrors()) {
            Map<String, String> errors = new HashMap<String, String>();
            for (Object errorObj : dataResult.getFieldErrors()) {
                FieldError error = (FieldError) errorObj;
                errors.put(error.getField(), facade.getCommonService().getMessage(error.getCode(), null, error.getArguments()));
            }
            uploadResponse.setDataErrors(errors);
        }

    }

    public File uploadFile(byte[] data) {
        File tempFile = facade.getFileService().createTemporaryFile("img_upload_" + Math.round(Math.random() * 100000));
        try {
            FileOutputStream output = new FileOutputStream(tempFile);
            IOUtils.write(data, output);
        } catch (IOException e) {
            return null;
        }
        return tempFile;
    }
}
