package com.imcode.imcms.addon.imagearchive.controller.web;

import com.imcode.imcms.addon.imagearchive.SessionConstants;
import com.imcode.imcms.addon.imagearchive.command.SearchImageCommand;
import com.imcode.imcms.addon.imagearchive.comparators.CategoryComparator;
import com.imcode.imcms.addon.imagearchive.comparators.KeywordComparator;
import com.imcode.imcms.addon.imagearchive.entity.*;
import com.imcode.imcms.addon.imagearchive.service.Facade;
import com.imcode.imcms.addon.imagearchive.util.*;
import com.imcode.imcms.addon.imagearchive.validator.SearchImageValidator;
import org.apache.commons.lang3.StringEscapeUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.validation.ValidationUtils;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.IOException;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

@Controller
public class SearchImageController {
	private static final Log log = LogFactory.getLog(SearchImageController.class);

	private static final Pattern LOCAL_PAGE_PATTERN = Pattern.compile("/client/archive/page/(\\d+)?");
    private static final Pattern PAGE_PATTERN = Pattern.compile("/archive/page/(\\d+)?");

    private static final String PAGINATION_KEY = Utils.makeKey(SearchImageController.class, "pagination");
	private static final String COMMAND_KEY = Utils.makeKey(SearchImageController.class, "command");

	@Autowired
	private Facade facade;

	private static int getPage(HttpServletRequest request) {
        final String requestURI = request.getRequestURI();

        Pattern pattern = (requestURI.startsWith("/archive"))
                ? PAGE_PATTERN
                : LOCAL_PAGE_PATTERN;

        Matcher matcher = pattern.matcher(requestURI);

		int page = 0;
		if (matcher.find()) {
			try {
				page = Integer.parseInt(matcher.group(1), 10);
				page = Math.max(page - 1, 0);
			} catch (NumberFormatException ignore) {
			}
		}

		return page;
	}

	// todo: next two methods are almost the same, remove code duplicates!!!1!

	@RequestMapping({"/archive", "/"})
	public ModelAndView indexHandler(
			@ModelAttribute("search") SearchImageCommand command,
			BindingResult result,
			@RequestParam(required = false) String returnTo,
			HttpServletRequest request,
            HttpServletResponse response,
			HttpSession session) throws IOException {

        User user = ClientUtils.getLoggedInUser(request);

        if (user == null) {
            ClientUtils.redirectToLogin(response);
            return null;
        }

		SearchImageCommand temporaryCommand = new SearchImageCommand();
		temporaryCommand.copyFrom(command);
		returnTo = StringUtils.trimToNull(returnTo);

		if (returnTo != null) {
			session.setAttribute(SessionConstants.IMCMS_RETURN_URL, StringEscapeUtils.unescapeHtml4(returnTo));
		}

		ModelAndView mav = new ModelAndView("search_image");
		Pagination pag = Pagination.getPagination(session, PAGINATION_KEY, SearchImageCommand.DEFAULT_PAGE_SIZE);

		if (request.getParameter("show") == null) {
			SearchImageCommand cmd = (SearchImageCommand) session.getAttribute(COMMAND_KEY);
			if (cmd != null) {
				command.copyFrom(cmd);
			}
		} else {
			session.setAttribute(COMMAND_KEY, command);
			pag.setCurrentPage(0);
		}

		if (command.isClear()) {
			command.copyFrom(new SearchImageCommand());
		}

		mav.addObject("search", command);

		Set<Category> categories = new HashSet<Category>();
		for (Role role : user.getRoles()) {
			categories.addAll(facade.getRoleService().getUsableRoleCategories(role.getId()));
		}

		List<Keyword> keywords = facade.getKeywordService().getKeywords();
        List<String> artists = new ArrayList<String>(facade.getExifService().getAllArtists());
        List<Category> sortedCategories = new ArrayList<Category>(categories);

        Collections.sort(keywords, new KeywordComparator());
        Collections.sort(artists);
        Collections.sort(sortedCategories, new CategoryComparator());

		mav.addObject("categories", sortedCategories);
		mav.addObject("keywords", keywords);
		mav.addObject("artists", artists);

		SearchImageValidator validator = new SearchImageValidator(categories);
		ValidationUtils.invokeValidator(validator, temporaryCommand, result);

		if (result.hasErrors()) {
			return mav;
		}

		session.setAttribute(COMMAND_KEY, command);

        pag.setPageSize(command.getResultsPerPage());
        final PageResult<Image> imagePageResult = facade.getImageService().searchImages(pag, temporaryCommand);
        final Collection<Image> images = imagePageResult.getResult();
        pag = imagePageResult.getPagination();

		mav.addObject("imageCount", pag.getAllElementsCount());
		mav.addObject("images", images);
		mav.addObject("pag", pag);

        mav.addObject("user", user);

		return mav;
	}

	@RequestMapping("/page/*")
	public ModelAndView pageHandler(HttpServletRequest request,
                                    @ModelAttribute("search") SearchImageCommand command,
                                    BindingResult result,
                                    HttpServletResponse response,
                                    HttpSession session) {

	    User user = ClientUtils.getLoggedInUser(request);

        if (user == null) {
            ClientUtils.redirectToLogin(response);
            return null;
        }

        command = (SearchImageCommand) session.getAttribute(COMMAND_KEY);

        if (command == null) {
            return new ModelAndView("redirect:/archive/");
        }

		Pagination pag = Pagination.getPagination(session, PAGINATION_KEY, SearchImageCommand.DEFAULT_PAGE_SIZE);
		pag.setCurrentPage(getPage(request));

		ModelAndView mav = new ModelAndView("search_image");
		mav.addObject("search", command);

		List<Keyword> keywords = facade.getKeywordService().getKeywords();
		Collections.sort(keywords, new KeywordComparator());

        List<String> artists = new ArrayList<String>(facade.getExifService().getAllArtists());
        Collections.sort(artists);

        mav.addObject("keywords", keywords);
		mav.addObject("artists", artists);

		Set<Category> categories = new HashSet<Category>();

        for (Role role : user.getRoles()) {
			categories.addAll(facade.getRoleService().getUsableRoleCategories(role.getId()));
		}

        List<Category> sortedCategories = new ArrayList<Category>(categories);
        Collections.sort(sortedCategories, new CategoryComparator());

		mav.addObject("categories", sortedCategories);

        SearchImageCommand temporaryCommand = new SearchImageCommand();
        temporaryCommand.copyFrom(command);

        SearchImageValidator validator = new SearchImageValidator(categories);
        ValidationUtils.invokeValidator(validator, temporaryCommand, result);

        if (result.hasErrors()) {
            return mav;
        }

        pag.setPageSize(command.getResultsPerPage());
        final PageResult<Image> imagePageResult = facade.getImageService().searchImages(pag, temporaryCommand);
        final Collection<Image> images = imagePageResult.getResult();
        pag = imagePageResult.getPagination();

        mav.addObject("imageCount", pag.getAllElementsCount());
		mav.addObject("images", images);
		mav.addObject("pag", pag);

        mav.addObject("user", user);

		return mav;
	}
}
