package com.imcode.imcms.addon.imagearchive.controller.web;

import com.imcode.imcms.addon.imagearchive.Config;
import com.imcode.imcms.addon.imagearchive.command.ChangeImageDataCommand;
import com.imcode.imcms.addon.imagearchive.command.ExternalFilesCommand;
import com.imcode.imcms.addon.imagearchive.command.ExternalFilesSaveImageCommand;
import com.imcode.imcms.addon.imagearchive.comparators.LibraryEntryComparator;
import com.imcode.imcms.addon.imagearchive.dto.LibrariesDto;
import com.imcode.imcms.addon.imagearchive.dto.LibraryEntryDto;
import com.imcode.imcms.addon.imagearchive.entity.*;
import com.imcode.imcms.addon.imagearchive.json.UploadResponse;
import com.imcode.imcms.addon.imagearchive.service.Facade;
import com.imcode.imcms.addon.imagearchive.service.file.LibrarySort;
import com.imcode.imcms.addon.imagearchive.tag.func.Functions;
import com.imcode.imcms.addon.imagearchive.util.ClientUtils;
import com.imcode.imcms.addon.imagearchive.util.Pagination;
import com.imcode.imcms.addon.imagearchive.util.Utils;
import com.imcode.imcms.addon.imagearchive.util.image.Format;
import com.imcode.imcms.addon.imagearchive.util.image.ImageInfo;
import com.imcode.imcms.addon.imagearchive.util.image.ImageOp;
import com.imcode.imcms.addon.imagearchive.validator.ChangeImageDataValidator;
import com.imcode.imcms.addon.imagearchive.validator.ExternalFilesValidator;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.validation.ValidationUtils;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.OutputStream;
import java.util.*;

@Controller
@RequestMapping("/external-files")
public class ExternalFilesController {
    private static final Log log = LogFactory.getLog(ExternalFilesController.class);

    private static final String LIBRARY_KEY = Utils.makeKey(ExternalFilesController.class, "library");
    private static final String SORT_KEY = Utils.makeKey(ExternalFilesController.class, "sortBy");
    private static final String PAGINATION_KEY = Utils.makeKey(ExternalFilesController.class, "pagination");
    private static final String IMAGE_KEY = Utils.makeKey(ExternalFilesController.class, "image");
    private static final String KEYWORDS_KEY = Utils.makeKey(ExternalFilesController.class, "keywords");
    private static final String IMAGE_KEYWORDS_KEY = Utils.makeKey(ExternalFilesController.class, "imageKeywords");
    private static final String FILE_NAMES_KEY = Utils.makeKey(ExternalFilesController.class, "fileNames");
    private static final int FIRST_PAGE = 0;

    @Autowired
    private Facade facade;

    @Autowired
    private Config config;

    private static LibrarySort getSortBy(HttpSession session) {
        LibrarySort sortBy = (LibrarySort) session.getAttribute(SORT_KEY);
        if (sortBy == null) {
            sortBy = LibrarySort.FILENAME;
            sortBy.setDirection(LibrarySort.DIRECTION.ASC);
            session.setAttribute(SORT_KEY, sortBy);
        }

        return sortBy;
    }

    @RequestMapping
    public ModelAndView indexHandler(@RequestParam(required = false) Integer page,
                                     HttpServletRequest request,
                                     HttpServletResponse response,
                                     HttpSession session) {

        if (ClientUtils.isUserNotLoggedIn(request)) {
            ClientUtils.redirectToLogin(response);
            return null;
        }

        ModelAndView mav = new ModelAndView("external_files/external_files");
        File libraryFile = config.getImcmsLibrariesPath();
        LibrariesDto rootLibrary = new LibrariesDto(libraryFile);

        LibrariesDto currentLibrary = getLibrary(session);
        LibrarySort sortBy = getSortBy(session);

        Pagination pagination = Pagination.getPagination(session, PAGINATION_KEY);
        page = (page == null) ? FIRST_PAGE : Math.max(page - 1, 0);

        pagination.setCurrentPage(page);

        List<File> files = facade.getFileService()
                .listLibraryEntries(currentLibrary);

        pagination.update(files.size());
        int lastFileIndex = pagination.getStartPosition() + pagination.getPageSize();
        int skip = pagination.getStartPosition();

        if (lastFileIndex > files.size()) {
            lastFileIndex = files.size();
        }

        files = files.subList(skip, lastFileIndex);

        List<LibraryEntryDto> libraryEntries = new ArrayList<LibraryEntryDto>(files.size());

        for (File file : files) {
            LibraryEntryDto entry = new LibraryEntryDto();
            entry.setFileName(file.getName());
            entry.setFileSize((int) file.length());
            entry.setLastModified(file.lastModified());
            entry.setImageInfo(ImageOp.getImageInfo(config, file));

            libraryEntries.add(entry);
        }
        Collections.sort(libraryEntries, new LibraryEntryComparator(sortBy));

        mav.addObject("pagination", pagination);
        mav.addObject("currentLibrary", currentLibrary);
        mav.addObject("library", rootLibrary);
        mav.addObject("libraryEntries", libraryEntries);
        mav.addObject("sortBy", sortBy);
        mav.addObject("externalFiles", new ExternalFilesCommand());

        return mav;
    }

    @RequestMapping("/library")
    public String changeLibraryHandler(
            @RequestParam(required = false) String path,
            HttpSession session) {

        if (path != null) {
            LibrariesDto library = new LibrariesDto(new File(path));
            session.setAttribute(LIBRARY_KEY, library);
        }

        return "redirect:/archive/external-files";
    }

    @RequestMapping("/sort")
    public String changeSortByHandler(
            @RequestParam(required = false) String sortBy,
            HttpSession session) {
        LibrarySort sort;
        if (sortBy != null && (sort = LibrarySort.findByName(sortBy)) != null) {
            session.setAttribute(SORT_KEY, sort);
        }

        return "redirect:/archive/external-files";
    }

    @RequestMapping("/process")
    public ModelAndView processHandler(@ModelAttribute("externalFiles") ExternalFilesCommand command,
                                       BindingResult result,
                                       HttpServletRequest request,
                                       HttpServletResponse response,
                                       HttpSession session) {
        User user = ClientUtils.getLoggedInUser(request);

        if (user == null) {
            ClientUtils.redirectToLogin(response);
            return null;
        }

        LibrariesDto library = getLibrary(session);
        if (library == null) {
            return new ModelAndView("redirect:/archive/external-files");
        }

        if (command.getActivate() != null) {
            String[] fileNames = command.getFileNames();
            if (fileNames == null || fileNames.length == 0) {
                return new ModelAndView("redirect:/archive/external-files");
            }

            session.setAttribute(FILE_NAMES_KEY, fileNames);
            ModelAndView mav = new ModelAndView("external_files/external_files");
            mav.addObject("activate", true);

            if (fileNames.length == 1) {
                Image image;
                boolean alreadyInArchive =
                        ClientUtils.isInArchive(facade.getFileService().getImageFileFromLibrary(library, fileNames[0])).size() > 0;
                if (alreadyInArchive || (image = activateImage(library, fileNames[0])) == null) {
                    ModelAndView tmp = new ModelAndView("redirect:/archive/external-files");
                    tmp.addObject("activateError", true);
                    if (alreadyInArchive) {
                        tmp.addObject("alreadyInArchive", true);
                    }
                    return tmp;
                }
                session.setAttribute(IMAGE_KEY, image);

                Set<Category> categories = new HashSet<Category>();
                for (Role role : user.getRoles()) {
                    categories.addAll(facade.getRoleService().getUsableRoleCategories(role.getId()));
                } //facade.getRoleService().findCategories(user, Role.ALL_PERMISSIONS);


                ChangeImageDataCommand changeData = new ChangeImageDataCommand();
                changeData.fromImage(image);
                mav.addObject("changeData", changeData);
                mav.addObject("categories", categories);

                List<Keyword> keywords = facade.getImageService().findAvailableKeywords(image.getId());
                List<Keyword> imageKeywords = facade.getImageService().findImageKeywords(image.getId());

                session.setAttribute(KEYWORDS_KEY, keywords);
                session.setAttribute(IMAGE_KEYWORDS_KEY, imageKeywords);

                mav.addObject("keywords", keywords);
                mav.addObject("imageKeywords", imageKeywords);
                mav.addObject("image", image);
                mav.addObject("format", Format.findFormat(image.getFormat()));
            } else {
                List<Object[]> tuples = new ArrayList<Object[]>(fileNames.length);
                boolean generalActivationErrorOccured = false;
                boolean oneOfTheImageIsAlreadyInArchive = false;

                for (String fileName : fileNames) {
                    try {
                        File imageFile = facade.getFileService().getImageFileFromLibrary(library, fileName);
                        ImageInfo imageInfo;
                        boolean alreadyInArchive = ClientUtils.isInArchive(imageFile).size() > 0;
                        oneOfTheImageIsAlreadyInArchive |= alreadyInArchive;
                        if (imageFile == null || (imageInfo = ImageOp.getImageInfo(config, imageFile)) == null
                                || alreadyInArchive) {
                            generalActivationErrorOccured |= true;
                            continue;
                        }

                        tuples.add(new Object[]{imageFile, imageInfo, fileName});
                    } catch (Exception ex) {
                        log.warn(ex.getMessage(), ex);
                    }
                }

                if (!tuples.isEmpty()) {
                    facade.getImageService().createImages(tuples);
                }

                if (generalActivationErrorOccured) {
                    ModelAndView tmp = new ModelAndView("redirect:/archive/external-files");
                    tmp.addObject("activateError", true);
                    if (oneOfTheImageIsAlreadyInArchive) {
                        tmp.addObject("alreadyInArchive", true);
                    }

                    return tmp;
                }

                return new ModelAndView("redirect:/archive/external-files");
            }

            return mav;
        }

        return new ModelAndView("redirect:/archive/external-files");
    }

    /* Called by uploadify for multi-file upload(images, zips or both).
     * Stores images and images from zip archives
     * One request, one file.
     * User is only allowed to upload images to personal folder
     * Single image upload redirects to a form where attributes can be changed, when multiple files are uploaded
     * no form is shown*/
    @RequestMapping("/upload")
    public void uploadHandler(ExternalFilesCommand externalFilesUpload,
                              BindingResult result,
                              HttpServletRequest request,
                              HttpServletResponse response,
                              HttpSession session) {
        UploadResponse uploadResponse = new UploadResponse();
        String contextPath = request.getContextPath();

        File libraryFile = config.getImcmsLibrariesPath();
        LibrariesDto library = new LibrariesDto(libraryFile);
        session.setAttribute(LIBRARY_KEY, library);

        ExternalFilesValidator validator = new ExternalFilesValidator(facade);
        ValidationUtils.invokeValidator(validator, externalFilesUpload, result);

        if (!result.hasErrors()) {
            boolean success;
            if (!validator.isZipFile()) {
                success = facade.getFileService().storeImageToLibrary(library, validator.getTempFile(), validator.getFileName());
            } else {
                success = facade.getFileService().storeZipToLibrary(library, validator.getTempFile());
            }

            if (!success) {
                result.rejectValue("file", "externalFiles.fileError");
            }
        }

        if (validator.getTempFile() != null) {
            facade.getFileService().deleteTemporaryFile(
                    validator.getTempFile());
        }

        List<String> errors = new ArrayList<String>();
        if (result.hasErrors()) {
            for (Object errorObj : result.getFieldErrors()) {
                FieldError error = (FieldError) errorObj;
                errors.add(facade.getMessageSource().getMessage(error.getCode(), error.getArguments(), request.getLocale()));
            }
            uploadResponse.setImageErrors(errors);
        } else {
            uploadResponse.setRedirectOnAllComplete(contextPath + "/archive/external-files");
        }

        Utils.writeJSON(uploadResponse, response);
    }

    private Image activateImage(LibrariesDto library, String fileName) {
        File imageFile = facade.getFileService().getImageFileFromLibrary(library, fileName);
        ImageInfo imageInfo;
        if (imageFile == null || (imageInfo = ImageOp.getImageInfo(config, imageFile)) == null) {
            return null;
        }

        try {
            return facade.getImageService().createImage(imageFile, imageInfo, fileName, Image.STATUS_ACTIVE);
        } catch (Exception ex) {
            log.warn(ex.getMessage(), ex);
        }

        return null;
    }

    @RequestMapping("/change")
    public ModelAndView changeHandler(@ModelAttribute("changeData") ChangeImageDataCommand changeData,
                                      BindingResult result,
                                      ExternalFilesSaveImageCommand saveCommand,
                                      HttpServletRequest request,
                                      HttpServletResponse response,
                                      HttpSession session) {
        User user = ClientUtils.getLoggedInUser(request);
        Image image = (Image) session.getAttribute(IMAGE_KEY);

        if (user == null) {
            ClientUtils.redirectToLogin(response);
            return null;

        } else if (image == null) {
            return new ModelAndView("redirect:/archive/external-files");
        }

        if (saveCommand.getCancel() != null) {
            facade.getImageService().deleteImage(image.getId());
            session.removeAttribute(IMAGE_KEY);
            session.removeAttribute(KEYWORDS_KEY);
            session.removeAttribute(IMAGE_KEYWORDS_KEY);

            return new ModelAndView("redirect:/archive/external-files");
        }

        ChangeImageDataValidator validator = new ChangeImageDataValidator();
        ValidationUtils.invokeValidator(validator, changeData, result);

        ModelAndView mav = new ModelAndView("external_files/external_files");
        mav.addObject("activate", true);
        mav.addObject("image", image);

        List<String> keywords = changeData.getKeywordNames();
        List<String> imageKeywords = changeData.getImageKeywordNames();
        session.setAttribute(KEYWORDS_KEY, keywords);
        session.setAttribute(IMAGE_KEYWORDS_KEY, imageKeywords);
        mav.addObject("keywords", keywords);
        mav.addObject("imageKeywords", imageKeywords);

        List<Category> categories = facade.getCategoryService().getCategories(image);
        mav.addObject("categories", categories);
        mav.addObject("imageCategories", categories);

        if (saveCommand.getRotateLeft() != null || saveCommand.getRotateRight() != null) {
            if (saveCommand.getRotateLeft() != null) {
                facade.getFileService().rotateImage(image.getId(), -90, false);
            } else {
                facade.getFileService().rotateImage(image.getId(), 90, false);
            }

            return mav;

        } else if (result.hasErrors()) {
            return mav;
        }

        changeData.toImage(image);

        try {
            facade.getImageService().updateData(image, changeData.getCategoryIds(), imageKeywords);
            final String[] fileNames = getFiles(session);

            if (fileNames != null) {
                final LibrariesDto library = getLibrary(session);
                for (String fileName : fileNames) {
                    facade.getFileService().deleteFileFromLibrary(library, fileName);
                }
            }

            if (saveCommand.getSaveActivate() != null) {
                session.removeAttribute(IMAGE_KEY);
                session.removeAttribute(KEYWORDS_KEY);
                session.removeAttribute(IMAGE_KEYWORDS_KEY);

                return new ModelAndView("redirect:/archive/external-files");
            } else if (saveCommand.getSaveUse() != null) {
                session.removeAttribute(IMAGE_KEY);
                session.removeAttribute(KEYWORDS_KEY);
                session.removeAttribute(IMAGE_KEYWORDS_KEY);

                return new ModelAndView("redirect:/archive/use?id=" + image.getId());
            } else if (saveCommand.getSaveImageCard() != null) {
                session.removeAttribute(IMAGE_KEY);
                session.removeAttribute(KEYWORDS_KEY);
                session.removeAttribute(IMAGE_KEYWORDS_KEY);

                return new ModelAndView("redirect:/archive/image/" + image.getId());
            }

        } catch (Exception ex) {
            log.fatal(ex.getMessage(), ex);

            return new ModelAndView("redirect:/archive/external-files");
        }

        return mav;
    }

    @RequestMapping("/preview")
    public ModelAndView previewHandler(
            @RequestParam(required = false) String path,
            @RequestParam(required = false) String name) {

        LibrariesDto library = null;
        ImageInfo imageInfo = null;
        if (path != null) {
            library = new LibrariesDto(new File(path));

            name = StringUtils.trimToNull(name);

            if (name != null) {
                File imageFile = facade.getFileService().getImageFileFromLibrary(library, name);
                if (imageFile != null) {
                    imageInfo = ImageOp.getImageInfo(config, imageFile);
                }
            }
        }

        Map<String, Object> model = new HashMap<String, Object>();
        model.put("user", null);
        model.put("library", library);
        model.put("imageInfo", imageInfo);
        model.put("name", name);

        return new ModelAndView("external_files/preview", model);
    }

    /* Used by small tooltip when moving cursor over filenames in library entry listByNamedParams */
    @RequestMapping("/preview-tooltip")
    public ModelAndView previewTooltipHandler(
            @RequestParam(required = false) String path,
            @RequestParam(required = false) String name) {

        LibrariesDto library = null;
        ImageInfo imageInfo = null;
        Integer fileSize = null;
        String categoryNamesNeededToUse = "";
        boolean noCategories = false;
        if (path != null) {
            library = new LibrariesDto(new File(path));

            name = StringUtils.trimToNull(name);

            if (name != null) {
                File imageFile = facade.getFileService().getImageFileFromLibrary(library, name);
                List<Image> images = ClientUtils.isInArchive(imageFile);
                if (images.size() > 0) {
                    for (Image image : images) {
                        /* we only need one case of missing categories */
                        if (!noCategories) {
                            noCategories = image.getCategories() == null || image.getCategories().size() == 0;
                        }

                        List<String> categoryNames = new ArrayList<String>();
                        for (Category cat : ClientUtils.getCategoriesRequiredToUse(image)) {
                            categoryNames.add(cat.getName());
                        }

                        categoryNamesNeededToUse += Functions.join(categoryNames, ",");
                    }
                }

                if (imageFile != null) {
                    fileSize = (int) imageFile.length();
                    imageInfo = ImageOp.getImageInfo(config, imageFile);
                }
            }
        }

        Map<String, Object> model = new HashMap<String, Object>();
        model.put("user", null);
        model.put("library", library);
        model.put("imageInfo", imageInfo);
        model.put("name", name);
        model.put("size", fileSize);
        model.put("categoryNamesNeededToUse", categoryNamesNeededToUse);
        model.put("noCategories", noCategories);

        return new ModelAndView("external_files/preview-tooltip", model);
    }

    @RequestMapping("/image")
    public String imageHandler(@RequestParam(required = false) String path,
                               @RequestParam(required = false) String name,
                               HttpServletRequest request,
                               HttpServletResponse response) {

        User user = ClientUtils.getLoggedInUser(request);
        if (user == null || path == null) {
            Utils.sendErrorCode(response, HttpServletResponse.SC_NOT_FOUND);
            return null;
        }

        LibrariesDto library = new LibrariesDto(new File(path));

        File imageFile = facade.getFileService().getImageFileFromLibrary(library, name);
        if (imageFile == null) {
            Utils.sendErrorCode(response, HttpServletResponse.SC_NOT_FOUND);
            return null;
        }

        try {
            byte[] data = new ImageOp(config).input(imageFile)
                    .outputFormat(Format.JPEG)
                    .processToByteArray();

            if (data == null) {
                Utils.sendErrorCode(response, HttpServletResponse.SC_NOT_FOUND);

                return null;
            }

            Utils.addNoCacheHeaders(response);
            response.setContentLength(data.length);
            response.setContentType("image/jpeg");

            OutputStream output = null;
            try {
                output = new BufferedOutputStream(response.getOutputStream());
                IOUtils.copy(new ByteArrayInputStream(data), output);
                output.flush();
            } finally {
                IOUtils.closeQuietly(output);
            }
        } catch (Exception ex) {
            Utils.sendErrorCode(response, HttpServletResponse.SC_NOT_FOUND);
            return null;
        }

        return null;
    }

    private LibrariesDto getLibrary(HttpSession session) {
        LibrariesDto library = (LibrariesDto) session.getAttribute(LIBRARY_KEY);
        if (library == null) {
            library = new LibrariesDto(config.getImcmsLibrariesPath());
            session.setAttribute(LIBRARY_KEY, library);
        }

        return library;
    }

    private String[] getFiles(HttpSession session) {
        return ((String[]) session.getAttribute(FILE_NAMES_KEY));
    }
}
