package com.imcode.imcms.addon.imagearchive.controller.web;

import com.imcode.imcms.addon.imagearchive.command.*;
import com.imcode.imcms.addon.imagearchive.entity.*;
import com.imcode.imcms.addon.imagearchive.repository.CategoryRolesRepository;
import com.imcode.imcms.addon.imagearchive.service.Facade;
import com.imcode.imcms.addon.imagearchive.service.exception.CategoryExistsException;
import com.imcode.imcms.addon.imagearchive.service.exception.KeywordExistsException;
import com.imcode.imcms.addon.imagearchive.util.ClientUtils;
import com.imcode.imcms.addon.imagearchive.util.Utils;
import com.imcode.imcms.addon.imagearchive.validator.*;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.Predicate;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.validation.ValidationUtils;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.util.*;

@Controller
public class PreferencesController {
    private static final Log log = LogFactory.getLog(PreferencesController.class);
    private static final String ROLE_KEY = Utils.makeKey(PreferencesController.class, "role");

    @Autowired
    private Facade facade;

    @Autowired
    private CategoryRolesRepository categoryRolesRepository;

    private static Role getRole(HttpSession session, List<Role> roles) {
        Role role = (Role) session.getAttribute(ROLE_KEY);
        if (role == null && roles != null) {
            role = roles.get(0);
            session.setAttribute(ROLE_KEY, role);
        }

        return role;
    }

    @RequestMapping("/preferences")
    public String indexHandler(
            @ModelAttribute PreferencesActionCommand actionCommand,

            @ModelAttribute("createCategory") CreateCategoryCommand createCategoryCommand,
            BindingResult createCategoryResult,

            @ModelAttribute("editCategory") EditCategoryCommand editCategoryCommand,
            BindingResult editCategoryResult,

            @ModelAttribute("createKeyword") CreateKeywordCommand createKeywordCommand,
            BindingResult createKeywordResult,

            @ModelAttribute("editKeyword") EditKeywordCommand editKeywordCommand,
            BindingResult editKeywordResult,

            @ModelAttribute("saveCategories") SaveRoleCategoriesCommand roleCategoriesCommand,
            HttpServletRequest request,
            HttpServletResponse response,
            HttpSession session,
            Map<String, Object> model) {

        User user = ClientUtils.getLoggedInUser(request);
        if (user == null) {
            ClientUtils.redirectToLogin(response);
            return null;

        } else if (!ClientUtils.canAccessPreferences(user)) {
            return "redirect:/archive";
        }

        List<Role> roles = facade.getRoleService().getAllRoles();
        // rewrite using method from service(JPA repo)
        Collections.sort(roles, new Comparator<Role>() {
            @Override
            public int compare(Role role1, Role role2) {
                return role1.getName().compareTo(role2.getName());
            }
        });

        Role role = getRole(session, roles);
        Role previousRole = (Role) session.getAttribute("previousRole");
        if (previousRole != null && !role.equals(previousRole)) {
            model.put("editingRoles", true);
        }
        session.setAttribute("previousRole", role);
        model.put("currentRole", role);

        if (actionCommand.isCreateCategory()) {
            processCreateCategory(createCategoryCommand, createCategoryResult);
            model.put("editingCategories", true);

        } else if (actionCommand.isEditCategory()) {
            processEditCategory(editCategoryCommand);
            model.put("editingCategories", true);

        } else if (actionCommand.isSaveCategory()) {
            processSaveCategory(editCategoryCommand, editCategoryResult);
            model.put("editingCategories", true);

        } else if (actionCommand.isRemoveCategory()) {
            processRemoveCategory(editCategoryCommand);
            model.put("editingCategories", true);

        } else if (actionCommand.isCreateKeyword()) {
            processCreateKeyword(createKeywordCommand, createKeywordResult);
            model.put("editingKeywords", true);

        } else if (actionCommand.isSaveKeyword()) {
            processSaveKeyword(editKeywordCommand, editKeywordResult);
            model.put("editingKeywords", true);

        } else if (actionCommand.isRemoveKeyword()) {
            processRemoveKeyword(editKeywordCommand);
            model.put("editingKeywords", true);

        } else if (actionCommand.isSaveRoleCategories()) {
            processSaveRoleCategories(roleCategoriesCommand, model);
            model.put("editingRoles", true);
        }

        List<CategoryRoles> categoryRoles = facade.getRoleService().getCategoryRolesByRole(role);
        model.put("categoryRoles", categoryRoles);

        Collection<CategoryType> categoryTypes = facade.getCategoryTypeService().getCategoryTypes();

        // get catTypes only associated with Image Archive
        CollectionUtils.filter(categoryTypes, new Predicate<CategoryType>() {
            public boolean evaluate(CategoryType categoryType) {
                return categoryType.isImageArchive();
            }
        });

        // get all categories id from types
        Collection<Long> typeCategoriesId = facade.getCategoryTypeService().getCategoryIdsOfTypes(categoryTypes);
        List<Category> typeCategories = facade.getCategoryService().getCategories(typeCategoriesId);

        model.put("categoryTypes", categoryTypes);
        model.put("categories", typeCategories);
        model.put("keywords", facade.getKeywordService().getKeywords());
        model.put("roles", roles);
        model.put("freeCategories", typeCategories);
        model.put("roleCategories", typeCategories);
        model.put("allCategories", typeCategories);

        return "preferences";
    }

    private void processCreateCategory(CreateCategoryCommand command, BindingResult result) {
        CreateCategoryValidator validator = new CreateCategoryValidator();
        ValidationUtils.invokeValidator(validator, command, result);

        if (!result.hasErrors()) {
            String categoryName = command.getCreateCategoryName();
            int categoryTypeId = command.getCreateCategoryType();

            try {
                facade.getCategoryService().createCategory(categoryName, categoryTypeId);

                command.setCreateCategoryName("");
                command.setCreateCategoryType(0);
            } catch (CategoryExistsException ex) {
                result.rejectValue("createCategoryName", "preferences.categoryExistsError");
            }
        }
    }

    private void processEditCategory(EditCategoryCommand command) {
        Category category = facade.getCategoryService().getCategory(Long.valueOf(command.getEditCategoryId()));
        if (category != null) {
            command.setShowEditCategory(true);
            command.setEditCategoryName(category.getName());
        }
    }

    private void processSaveCategory(EditCategoryCommand command, BindingResult result) {
        EditCategoryValidator validator = new EditCategoryValidator();
        ValidationUtils.invokeValidator(validator, command, result);

        if (!result.hasErrors()) {
            long categoryId = command.getEditCategoryId();
            String newName = command.getEditCategoryName();

            try {
                facade.getCategoryService().updateCategory(categoryId, newName);

            } catch (CategoryExistsException ex) {
                result.rejectValue("editCategoryName", "preferences.categoryExistsError");
            }
        }
    }

    private void processRemoveCategory(EditCategoryCommand command) {
        final int categoryId = command.getEditCategoryId();
        final List<CategoryRoles> categoryRoles = categoryRolesRepository.findByCategoryId(categoryId);

        facade.getCategoryService().deleteCategory((long) categoryId);
        categoryRolesRepository.delete(categoryRoles);

        command.setEditCategoryName("");
        command.setShowEditCategory(false);
    }

    private void processCreateKeyword(CreateKeywordCommand command, BindingResult result) {
        CreateKeywordValidator validator = new CreateKeywordValidator();
        ValidationUtils.invokeValidator(validator, command, result);

        if (!result.hasErrors()) {
            String keywordName = command.getCreateKeywordName().toLowerCase();

            try {
                facade.getKeywordService().createKeyword(keywordName);

                command.setCreateKeywordName("");
            } catch (KeywordExistsException ex) {
                result.rejectValue("createKeywordName", "preferences.keywordExistsError");
            }
        }
    }

    private void processSaveKeyword(EditKeywordCommand command, BindingResult result) {
        EditKeywordValidator validator = new EditKeywordValidator();
        ValidationUtils.invokeValidator(validator, command, result);

        if (!result.hasErrors()) {
            long keywordId = command.getEditKeywordId();
            String KeywordName = command.getEditKeywordName();

            try {
                facade.getKeywordService().updateKeyword(keywordId, KeywordName);

            } catch (KeywordExistsException ex) {
                result.rejectValue("editKeywordName", "preferences.keywordExistsError");
            }
        }
    }

    private void processRemoveKeyword(EditKeywordCommand command) {
        facade.getKeywordService().deleteKeyword(command.getEditKeywordId());

        command.setEditKeywordName("");
    }

    private void processSaveRoleCategories(SaveRoleCategoriesCommand command, Map<String, Object> model) {
        try {
            Role role = (Role) model.get("currentRole");

            facade.getRoleService().assignCategoryRoles(role, command.getAssignedCategoryIds());
        } catch (Exception ex) {
            log.fatal(ex.getMessage(), ex);
        }
    }

    @RequestMapping("/preferences/role")
    public String changeCurrentRoleHandler(@RequestParam(required = false) Integer id,
                                           HttpServletRequest request,
                                           HttpServletResponse response,
                                           HttpSession session) {
        User user = ClientUtils.getLoggedInUser(request);
        if (user == null) {
            ClientUtils.redirectToLogin(response);
            return null;

        } else if (!ClientUtils.canAccessPreferences(user)) {
            return "redirect:/archive";
        }

        Role role;
        if (id != null && (role = facade.getRoleService().getRoleById(id)) != null) {
            session.setAttribute(ROLE_KEY, role);
        }

        return "redirect:/archive/preferences";
    }
}
