package com.imcode.imcms.addon.imagearchive.util;

import com.imcode.imcms.addon.imagearchive.Config;
import com.imcode.imcms.addon.imagearchive.SessionConstants;
import com.imcode.imcms.addon.imagearchive.entity.Category;
import com.imcode.imcms.addon.imagearchive.entity.Image;
import com.imcode.imcms.addon.imagearchive.entity.Role;
import com.imcode.imcms.addon.imagearchive.entity.User;
import com.imcode.imcms.addon.imagearchive.service.Facade;
import org.apache.commons.beanutils.BeanMap;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.servlet.ServletRequest;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

@Component
public class ClientUtils {
    private static final Log log = LogFactory.getLog(ClientUtils.class);

    private static Facade facade;

    @Autowired
    private ClientUtils(Facade facade1) {
        facade = facade1;
    }

    public static Facade getFacade() {
        return facade;
    }

    public static User getLoggedInUser(HttpServletRequest request) {
        HttpSession session = request.getSession(false);

        if (session == null) {
            return null;
        }

        User user = getUserFromSession(session);
        Long userId = getUserIdFromRequest(request);

        if (shouldUserBeFounded(user, userId)) {
            user = getFacade().getUserService().findUserById(userId);

            if (isUserLoggedIn(user)) {
                saveUserToSession(session, user);
            }
        }

        return (isUserLoggedIn(user) ? user : null);
    }

    private static Long getUserIdFromRequest(ServletRequest request) {
        Object cms = request.getAttribute(SessionConstants.CONTENT_MANAGEMENT_SYSTEM_REQUEST_ATTRIBUTE);

        Long userId = 1L;
//        Long userId = null;
        // do not forget to change to userId = null when commit or deploy !!!

        if (cms != null) {
            BeanMap introspected = new BeanMap(cms);
            if (introspected.containsKey("currentUser")) {
                BeanMap objectUser = new BeanMap(introspected.get("currentUser"));
                userId = Long.valueOf(ObjectUtils.toString(objectUser.get("id"), "2"));
            }
        }

        return userId;
    }

    private static boolean shouldUserBeFounded(User user, Long userId) {
        return ((userId != null) && ((user == null) || (userId.compareTo(user.getId()) != 0)));
    }

    private static void saveUserToSession(HttpSession session, User user) {
        session.setAttribute(SessionConstants.USER, user);
    }

    public static User getUserFromSession(HttpSession session) {
        return (User) session.getAttribute(SessionConstants.USER);
    }

    public static String getImcmsReturnToUrl(HttpSession session) {
        return (String) session.getAttribute(SessionConstants.IMCMS_RETURN_URL);
    }

    public static boolean isUserNotLoggedIn(HttpServletRequest request) {
        User user = getLoggedInUser(request);
        return isUserNotLoggedIn(user);
    }

    public static boolean isUserNotLoggedIn(User user) {
        return (user == null || user.isDefault());
    }

    public static boolean isUserLoggedIn(User user) {
        return !isUserNotLoggedIn(user);
    }

    public static void redirectToLogin(HttpServletResponse response) {
        try {
            Config config = getFacade().getConfig();
            response.sendRedirect(config.getImcmsRootUrl() + "/" + config.getImcmsLoginUrlPath());
        } catch (IOException ex) {
            log.warn(ex.getMessage(), ex);
        }
    }

    public static List<Image> isInArchive(File img) {
        List<Image> sameImages = new ArrayList<Image>();
        List<Image> archiveImages = getFacade().getImageService().getAllImages();
        for (Image image : archiveImages) {
            if (img != null && image.getName().equals(img.getName()) && image.getFileSize() == img.length()) {
                sameImages.add(image);
            }
        }

        return sameImages;
    }

    public static Collection<Category> getCategoriesRequiredToUse(Image img) {
        final Facade localFacade = getFacade();
        List<Category> userCategories = localFacade.getCategoryService().getCategories();
        List<Category> imageCategories = localFacade.getCategoryService().getCategories(img);
        Collection<Category> categoriesUserCantUse = Collections.emptyList();
        if (imageCategories != null) {
            categoriesUserCantUse = CollectionUtils.subtract(imageCategories, userCategories);
        }

        return categoriesUserCantUse;
    }

    public static boolean canAccessPreferences(User user) {
        return user.isSuperadmin() || isImageAdmin(user);
    }

    public static boolean isImageAdmin(User user) {
        Role role = getFacade().getRoleService().getRoleByName("Bildadmin");
        return role != null && user.hasRole(role);
    }

    public static boolean canUseOrChangeImage(Image image, User user) {
        return canChangeImage(image, user) || canUseImage(image, user);
    }

    public static boolean canUseImage(Image image, User user) {
        return (user.isSuperadmin()
                || isImageAdmin(user)
                || image.getCategories().size() == 0
                || isImageOfUsableRoleCategories(image, user)
        );
    }

    private static boolean isImageOfUsableRoleCategories(Image image, User user) {
        final List<Category> usableRoleCategories = getFacade().getRoleService()
                .getUsableRoleCategories(user.getRoles().toArray(new Role[]{}));

        return hasImageNextCategories(image, usableRoleCategories);
    }

    private static boolean hasImageNextCategories(Image image, List<Category> categories) {
        List<Long> categoryIds = new ArrayList<Long>();

        for (Category category : categories) {
            categoryIds.add(category.getId());
        }

        final List<Long> imageCategories = image.getCategories();

        return (CollectionUtils.containsAny(categoryIds, imageCategories));
    }

    public static boolean canChangeImage(Image image, User user) {
        final boolean canChangeImage = (user.isSuperadmin()
                || isImageAdmin(user)
                || image.getCategories().size() == 0
                || isImageOfChangeableRoleCategories(image, user)
        );

        image.setCanChange(canChangeImage);
        return image.isCanChange();
    }

    private static boolean isImageOfChangeableRoleCategories(Image image, User user) {
        final List<Category> usableRoleCategories = getFacade().getRoleService()
                .getChangeableRoleCategories(user.getRoles().toArray(new Role[]{}));

        return hasImageNextCategories(image, usableRoleCategories);
    }
}
