package com.imcode.imcms.addon.imagearchive.validator;

import com.imcode.imcms.addon.imagearchive.command.SearchImageCommand;
import com.imcode.imcms.addon.imagearchive.entity.Category;
import com.imcode.imcms.addon.imagearchive.tag.func.Functions;
import com.imcode.imcms.addon.imagearchive.util.ValidatorUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.validation.Errors;
import org.springframework.validation.Validator;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;


public class SearchImageValidator implements Validator {
	private Collection<Category> availableCategories;

	public SearchImageValidator(Collection<Category> categories) {
		availableCategories = categories;
	}

	public boolean supports(Class clazz) {
		return SearchImageCommand.class.isAssignableFrom(clazz);
	}

	public void validate(Object target, Errors errors) {
		SearchImageCommand command = (SearchImageCommand) target;

		short show = command.getShow();
		if (show < SearchImageCommand.SHOW_ALL || show > SearchImageCommand.SHOW_WITH_VALID_LICENCE) {
			command.setShow(SearchImageCommand.SHOW_ALL);
		}

		List<Integer> availableCategoryIds = new ArrayList<Integer>();
		for (Category category : availableCategories) {
			if (null != category) {
				availableCategoryIds.add(category.getId().intValue());
			}
		}
		availableCategoryIds.add(-2);
		if (command.getCategoryIds().contains(-1)) {
			command.setCategoryIds(availableCategoryIds);
		} else {
			List<Integer> categoryIds = new ArrayList<Integer>();
			List<Integer> unavailableCategoryIds = new ArrayList<Integer>();
			for (int categoryId : command.getCategoryIds()) {
				if (availableCategoryIds.contains(categoryId)) {
					categoryIds.add(categoryId);
				} else {
					unavailableCategoryIds.add(categoryId);
				}
			}
			if (unavailableCategoryIds.size() > 0) {
				errors.rejectValue("categoryIds", "searchImage.categoryPermissionError", new String[]{Functions.join(unavailableCategoryIds, ", ")}, null);
			}
			command.setCategoryIds(categoryIds);
		}

		ValidatorUtils.rejectValueIfLonger("freetext", 120, "fieldLengthError", errors);
		command.setFreetext(StringUtils.trimToNull(command.getFreetext()));

		setDates(command, errors);

		int resultsPerPage = command.getResultsPerPage();
		if (resultsPerPage < 10 || resultsPerPage > 100) {
			command.setResultsPerPage(SearchImageCommand.DEFAULT_PAGE_SIZE);
		}

		short sortBy = command.getSortBy();
		if (sortBy < SearchImageCommand.SORT_BY_ARTIST || sortBy > SearchImageCommand.SORT_BY_ENTRY_DATE) {
			command.setSortBy(SearchImageCommand.SORT_BY_ARTIST);
		}

		String artist = StringUtils.trimToNull(command.getArtist());
		command.setArtist(artist);
	}

	private Date parseDate(String date) throws ParseException {
		SimpleDateFormat df = new SimpleDateFormat("yyyy-MM-dd");
		String utilDate = StringUtils.trimToNull(date);
		if (utilDate != null) {
			return df.parse(utilDate);
		} else return null;
	}

	private void setDates(SearchImageCommand command, Errors errors) {
		try {
			command.setLicenseDate(parseDate(command.getLicenseDt()));
		} catch (ParseException ex) {
			errors.rejectValue("licenseDt", "invalidStartDateError");
		}

		try {
			command.setLicenseEndDate(parseDate(command.getLicenseEndDt()));
		} catch (ParseException ex) {
			errors.rejectValue("licenseEndDt", "invalidEndDateError");
		}

		try {
			command.setActiveDate(parseDate(command.getActiveDt()));
		} catch (ParseException ex) {
			errors.rejectValue("activeDt", "invalidStartDateError");
		}

		try {
			command.setActiveEndDate(parseDate(command.getActiveEndDt()));
		} catch (ParseException ex) {
			errors.rejectValue("activeEndDt", "invalidEndDateError");
		}
	}
}
