package com.imcode.imcms.addon.imagearchive.service.impl;

import com.imcode.imcms.addon.imagearchive.entity.ClientKeyword;
import com.imcode.imcms.addon.imagearchive.repository.ClientKeywordsRepository;
import com.imcode.imcms.addon.imagearchive.service.ClientKeywordService;
import com.imcode.imcms.addon.imagearchive.service.exception.KeywordExistsException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;

@Service
@Transactional
public class ClientKeywordServiceImpl implements ClientKeywordService {
    @Autowired
    private ClientKeywordsRepository keywordsRepository;

    public ClientKeyword createKeyword(final String newKeywordName) throws KeywordExistsException {
        if (existsKeyword(newKeywordName)) {
            throw new KeywordExistsException();
        }

        ClientKeyword keyword = new ClientKeyword();
        keyword.setName(newKeywordName);
        keywordsRepository.save(keyword);
        return keyword;
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public List<ClientKeyword> getKeywords() {
        List<ClientKeyword> keywords = keywordsRepository.findAll();
        return keywords;
    }

    @Transactional
    public void deleteKeyword(final Long keywordId) {
        if (!keywordsRepository.exists(keywordId)) return;
        removeKeywordImages(keywordId);
        try {
            keywordsRepository.delete(keywordId);
        } catch (Exception ignore) {
            ClientKeyword keyword = keywordsRepository.getOne(keywordId);
            keywordsRepository.save(keyword);
        } finally {
            try {
                keywordsRepository.flush();
            } catch (Exception ignore) {
                // only now keyword should be deleted
            }
        }
    }

    @Transactional
    private void removeKeywordImages(Long keywordId) {
        ClientKeyword keyword = keywordsRepository.getOne(keywordId);
        keyword.getImages().clear();
        keywordsRepository.save(keyword);
        keywordsRepository.flush();
    }

    public void updateKeyword(final Long keywordId, final String keywordName) throws KeywordExistsException {
        ClientKeyword keyword = keywordsRepository.findOne(keywordId);
        if (keyword != null) {
            keyword.setName(keywordName);
            keywordsRepository.save(keyword);
        }
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public boolean existsKeyword(final String newKeywordName) {
        return keywordsRepository.findByName(newKeywordName) != null;
    }

    public List<ClientKeyword> findKeywordsByNameLike(String text) {
        return keywordsRepository.findByNameLike(text);
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public ClientKeyword getKeyword(Long id) {
        return keywordsRepository.findOne(id);
    }

    @Transactional(propagation = Propagation.SUPPORTS, readOnly = true)
    public ClientKeyword findByName(String name) {
        return keywordsRepository.findByName(name);
    }

    public ClientKeyword createOrGet(String name) {
        try {
            return createKeyword(name);
        } catch (KeywordExistsException e) {
            return findByName(name);
        }
    }

    public List<ClientKeyword> findByName(List<String> names) {
        List<ClientKeyword> keywords = new ArrayList<ClientKeyword>();

        for (String name : names) {
            ClientKeyword keyword = findByName(name);

            if (keyword != null) {
                keywords.add(keyword);
            }
        }

        return keywords;
    }
}
