package com.imcode.imcms.addon.imagearchive.controller.web;

import com.imcode.imcms.addon.imagearchive.Config;
import com.imcode.imcms.addon.imagearchive.command.ChangeImageDataCommand;
import com.imcode.imcms.addon.imagearchive.command.ExportImageCommand;
import com.imcode.imcms.addon.imagearchive.command.ImageCardChangeActionCommand;
import com.imcode.imcms.addon.imagearchive.editors.EnumEditor;
import com.imcode.imcms.addon.imagearchive.entity.*;
import com.imcode.imcms.addon.imagearchive.service.Facade;
import com.imcode.imcms.addon.imagearchive.util.ClientUtils;
import com.imcode.imcms.addon.imagearchive.util.SessionUtils;
import com.imcode.imcms.addon.imagearchive.util.Utils;
import com.imcode.imcms.addon.imagearchive.util.exif.ExifData;
import com.imcode.imcms.addon.imagearchive.util.exif.ExifUtils;
import com.imcode.imcms.addon.imagearchive.util.exif.Flash;
import com.imcode.imcms.addon.imagearchive.util.image.*;
import com.imcode.imcms.addon.imagearchive.validator.ChangeImageDataValidator;
import com.imcode.imcms.addon.imagearchive.validator.ExportImageValidator;
import com.imcode.imcms.addon.imagearchive.validator.ImageUploadValidator;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.validation.ValidationUtils;
import org.springframework.web.bind.ServletRequestDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.*;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


@Controller
public class ImageCardController {
    private static final Log log = LogFactory.getLog(ImageCardController.class);

    private static final Pattern IMAGE_ID_PATTERN = Pattern.compile("/archive/image/([^/]+)/?");
    private static final String IMAGE_KEY = Utils.makeKey(ImageCardController.class, "image");

    @Autowired
    private Facade facade;

    @Autowired
    private Config config;

    @InitBinder
    protected void initBinder(HttpServletRequest request, ServletRequestDataBinder binder) {
        binder.registerCustomEditor(ExportImageCommand.SizeUnit.class, new EnumEditor());
    }

    @RequestMapping({"/image/*", "/image/*/"})
    public ModelAndView indexHandler(
            @ModelAttribute("exportImage") ExportImageCommand command,
            BindingResult result,
            HttpServletRequest request,
            HttpServletResponse response,
            HttpSession session) {

        User user = SessionUtils.getUser(request, session, facade);

        Long imageId = getImageId(request);
        Image image;

        if (imageId == null || (image = facade.getImageService().findById(imageId)) == null
                || !ClientUtils.canUseOrChangeImage(image, user, facade)) {
            return new ModelAndView("redirect:/archive/");
        }


        if (command.getExport() != null && !image.isArchived() && ClientUtils.canUseOrChangeImage(image, user, facade)) {
            ExportImageValidator validator = new ExportImageValidator();
            ValidationUtils.invokeValidator(validator, command, result);

            Format imageFormat = Format.findFormat(command.getFileFormat());

            if (imageFormat != null && imageFormat.isWritable() && !result.hasErrors()) {
                if (processExport(imageId, image, imageFormat, command, response)) {
                    return null;
                }
            }
        } else {

            Format format = Format.findFormat(image.getFormat());
            if (format.isWritable()) {
                command.setFileFormat(format.getOrdinal());
            }

            command.setWidth(image.getWidth());
            command.setHeight(image.getHeight());
        }

// FIXME: add new extra table imageMetaIds
        //  facade.getImageService().setImageMetaIds(image);

        ModelAndView mav = new ModelAndView("image_card/image_card");
        mav.addObject("image", image);
        mav.addObject("categories", getCategories(image));
        mav.addObject("keywords", getKeywords(image));
        mav.addObject("canUseInImcms", SessionUtils.getImcmsReturnToUrl(request.getSession()) != null
                && ClientUtils.canUseOrChangeImage(image, user, facade));
        mav.addObject("format", Format.findFormat(image.getFormat()));
        mav.addObject("canExport", ClientUtils.canUseOrChangeImage(image, user, facade));

        return mav;
    }

    @RequestMapping("/image/*/exif")
    public ModelAndView exifHandler(@ModelAttribute("exportImage") ExportImageCommand command,
                                    HttpServletRequest request, HttpSession session) {
        User user = SessionUtils.getUser(request, session, facade);

        Long imageId = getImageId(request);
        Image image;
        if (imageId == null || (image = facade.getImageService().findById(imageId)) == null
                || !ClientUtils.canUseOrChangeImage(image, user, facade)) {
            return new ModelAndView("redirect:/archive");
        }

        Format format = Format.findFormat(image.getFormat());
        if (format.isWritable()) {
            command.setFileFormat(format.getOrdinal());
        }

        command.setWidth(image.getWidth());
        command.setHeight(image.getHeight());

        Exif originalExif = facade.getImageService().findExifByPK(image.getId());
        image.setExif(originalExif);
//        image.setOriginalExif(originalExif);

        ModelAndView mav = new ModelAndView("image_card/image_card");
        mav.addObject("action", "exif");
        mav.addObject("image", image);
        mav.addObject("canExport", ClientUtils.canUseOrChangeImage(image, user, facade));
        mav.addObject("canUseInImcms", SessionUtils.getImcmsReturnToUrl(request.getSession()) != null
                && (ClientUtils.canUseOrChangeImage(image, user, facade)));

        return mav;
    }

    /* Doesn't actually delete images, as marks them as erased(unavailable) */
    @RequestMapping("/image/*/erase")
    public ModelAndView eraseHandler(
            HttpServletRequest request, HttpServletResponse response, HttpSession session) {
        User user = SessionUtils.getUser(request, session, facade);
        if (user == null) {
            ClientUtils.redirectToLogin(response, facade);

            return null;
        }

        Long imageId = getImageId(request);
        Image image = facade.getImageService().findById(imageId);
        image.setCanChange(ClientUtils.canChangeImage(image, user, facade));
        if (imageId == null || image == null) {
            return new ModelAndView("redirect:/archive/");
        } else if (image.isArchived() || !image.isCanChange()) {
            return new ModelAndView("redirect:/archive/image/" + imageId);
        }

        facade.getImageService().archiveImage(imageId);
        return new ModelAndView("redirect:/archive/");
    }

    @RequestMapping("/image/*/change")
    public ModelAndView changeHandler(
            @ModelAttribute("changeData") ChangeImageDataCommand changeData,
            BindingResult result,
            @ModelAttribute ImageCardChangeActionCommand action,
            HttpServletRequest request,
            @RequestParam(required = false) Boolean redirectToImageCard, HttpSession session) {
        User user = SessionUtils.getUser(request, session, facade);

        if (user == null) {
            return new ModelAndView("redirect:/archive/");
        }

        Long imageId = getImageId(request);

        if (imageId == null) {
            return new ModelAndView("redirect:/archive/");
        }

        Set<Category> categories = new HashSet<Category>();
        for (Role role : user.getRoles()) {
            categories.addAll(facade.getRoleService().getUsableRoleCategories(role.getId()));
        }

        ModelAndView mav = new ModelAndView("image_card/image_card");
        mav.addObject("action", "change");

        if (!action.isSet()) {
            Image image = facade.getImageService().findById(imageId);

            image.setCanChange(ClientUtils.canChangeImage(image, user, facade));

            if (image == null) {
                return new ModelAndView("redirect:/archive/");
            } else if (image.isArchived() || !image.isCanChange()) {
                return new ModelAndView("redirect:/archive/image/" + imageId);
            }

            session.setAttribute(IMAGE_KEY, image);
            changeData.fromImage(image);
            mav.addObject("image", image);
            mav.addObject("format", Format.findFormat(image.getFormat()));

            facade.getFileService().createTemporaryCopyOfCurrentImage(image.getId());
            List<Category> imageCategory = facade.getImageService().findImageCategories(image.getId());


            mav.addObject("categories", getAvailableCategories(categories, imageCategory));
            mav.addObject("imageCategories", imageCategory);

            List<Keyword> keywords = facade.getImageService().findAvailableKeywords(image.getId());
            List<Keyword> imageKeywords = facade.getImageService().findImageKeywords(image.getId());
            mav.addObject("keywords", keywords);
            mav.addObject("imageKeywords", imageKeywords);
        } else {
            Image image = (Image) session.getAttribute(IMAGE_KEY);
            image.setCanChange(ClientUtils.canChangeImage(image, user, facade));

            if (image == null) {
                return new ModelAndView("redirect:/archive/");
            } else if (!image.isCanChange()) {
                return new ModelAndView("redirect:/archive/image/" + imageId);
            }
            mav.addObject("image", image);
            mav.addObject("format", Format.findFormat(image.getFormat()));

            if (action.isCancel()) {
                session.removeAttribute(IMAGE_KEY);

                facade.getFileService().deleteTemporaryImage(imageId);

                return new ModelAndView("redirect:/archive/image/" + image.getId());
            }

            List<String> keywordNames = changeData.getKeywordNames();
            List<String> imageKeywordNames = changeData.getImageKeywordNames();

            List<Category> imageCategory = facade.getImageService().findImageCategories(image.getId());

            List<Keyword> keywords = facade.getKeywordService().findByName(keywordNames);
            List<Keyword> imageKeywords = facade.getKeywordService().findByName(imageKeywordNames);

            mav.addObject("keywords", keywords);
            mav.addObject("imageKeywords", imageKeywords);
            mav.addObject("categories", getAvailableCategories(categories, imageCategory));
            mav.addObject("imageCategories", imageCategory);

            if (action.isUpload()) {
                ImageUploadValidator validator = new ImageUploadValidator(facade);
                ValidationUtils.invokeValidator(validator, changeData.getFile(), result);

                String copyright = "";
                String description = "";
                String artist = "";
                String manufacturer = null;
                String model = null;
                String compression = null;
                Double exposure = null;
                String exposureProgram = null;
                Float fStop = null;
                Date dateOriginal = null;
                Date dateDigitized = null;
                Flash flash = null;
                Float focalLength = null;
                String colorSpace = null;
                Integer xResolution = null;
                Integer yResolution = null;
                Integer resolutionUnit = null;
                Integer pixelXDimension = null;
                Integer pixelYDimension = null;
                Integer ISO = null;

                File tempFile = validator.getTempFile();
                ExifData data = ExifUtils.getExifData(tempFile);
                if (data != null) {
                    copyright = StringUtils.substring(data.getCopyright(), 0, 255);
                    description = StringUtils.substring(data.getDescription(), 0, 255);
                    artist = StringUtils.substring(data.getArtist(), 0, 255);
                    xResolution = data.getxResolution();
                    yResolution = data.getyResolution();
                    manufacturer = data.getManufacturer();
                    model = data.getModel();
                    compression = data.getCompression();
                    exposure = data.getExposure();
                    exposureProgram = data.getExposureProgram();
                    fStop = data.getfStop();
                    flash = data.getFlash();
                    focalLength = data.getFocalLength();
                    colorSpace = data.getColorSpace();
                    resolutionUnit = data.getResolutionUnit();
                    pixelXDimension = data.getPixelXDimension();
                    pixelYDimension = data.getPixelYDimension();
                    ISO = data.getISO();
                    dateOriginal = data.getDateOriginal();
                    dateDigitized = data.getDateDigitized();
                }
                int fileSize = (int) tempFile.length();

                if (!facade.getFileService().storeImage(tempFile, imageId, true)) {
                    return mav;
                }

                changeData.setChangedFile(true);

                image.setFileSize(fileSize);

//                Exif changedExif = new Exif(xResolution, yResolution, description, artist, copyright, Exif.TYPE_CHANGED,
//                        manufacturer, model, compression, exposure, exposureProgram, fStop, flash, focalLength, colorSpace,
//                        resolutionUnit, pixelXDimension, pixelYDimension, dateOriginal, dateDigitized, ISO);
                Exif originalExif = new Exif(xResolution, yResolution, description, artist, copyright,
                        manufacturer, model, compression, exposure, exposureProgram, fStop, flash, focalLength, colorSpace,
                        resolutionUnit, pixelXDimension, pixelYDimension, dateOriginal, dateDigitized, ISO);
//                image.setChangedExif(changedExif);
                image.setExif(originalExif);
//                image.setOriginalExif(originalExif);
                image.setName(StringUtils.substring(validator.getImageName(), 0, 255));

//                String uploadedBy = String.format("%s %s", user.getFirstName(), user.getLastName()).trim();
                String uploadedBy = "";
                image.setUploadedBy(uploadedBy);

                ImageInfo imageInfo = validator.getImageInfo();

                image.setFormat(imageInfo.getFormat().getOrdinal());

                image.setWidth(imageInfo.getWidth());
                image.setHeight(imageInfo.getHeight());

                changeData.fromImage(image);

                return mav;
            }

            ChangeImageDataValidator validator = new ChangeImageDataValidator(facade);
            ValidationUtils.invokeValidator(validator, changeData, result);

            if (action.getRotateLeft() != null) {
                facade.getFileService().rotateImage(image.getId(), -90, true);
                changeData.setRotation(changeData.getRotation() - 90);
                if (changeData.getRotation() <= -360) {
                    changeData.setRotation(0);
                }

            } else if (action.getRotateRight() != null) {
                facade.getFileService().rotateImage(image.getId(), 90, true);
                changeData.setRotation(changeData.getRotation() + 90);
                if (changeData.getRotation() >= 360) {
                    changeData.setRotation(0);
                }

            } else if (!result.hasErrors()) {
                changeData.toImage(image);

                if (changeData.isChangedFile()) {
                    facade.getImageService().updateFullData(image, changeData.getCategoryIds(), imageKeywordNames);
                    changeData.setChangedFile(false);

                } else {
                    facade.getImageService().updateData(image, changeData.getCategoryIds(), imageKeywordNames);
                }
                imageCategory = facade.getImageService().findImageCategories(image.getId());
                /* refreshing categories since those are also set earlier, before update(in case of file upload) */
                mav.getModel().remove("categories");
                mav.getModel().remove("imageCategories");
                mav.addObject("categories", getAvailableCategories(categories, imageCategory));
                mav.addObject("imageCategories", imageCategory);

                facade.getFileService().copyTemporaryImageToCurrent(imageId);

                if (action.isUse()) {
                    facade.getFileService().deleteTemporaryImage(image.getId());
                    session.removeAttribute(IMAGE_KEY);

                    return new ModelAndView("redirect:/archive/use?id=" + image.getId());
                } else if (action.isImageCard()) {
                    facade.getFileService().deleteTemporaryImage(image.getId());
                    session.removeAttribute(IMAGE_KEY);

                    return new ModelAndView("redirect:/archive/image/" + image.getId());
                } else if (redirectToImageCard != null && redirectToImageCard) {
                    facade.getFileService().deleteTemporaryImage(image.getId());
                    session.removeAttribute(IMAGE_KEY);

                    return new ModelAndView("redirect:/archive/image/" + image.getId());
                }
            }
        }

        return mav;
    }

    private static Long getImageId(HttpServletRequest request) {
        String uri = request.getRequestURI();

        Matcher matcher = IMAGE_ID_PATTERN.matcher(uri);
        if (matcher.find()) {
            try {
                return Long.parseLong(matcher.group(1), 10);
            } catch (Exception ex) {
            }
        }

        return null;
    }

    @RequestMapping("/image/*/unarchive")
    public ModelAndView unarchiveHandler(HttpServletRequest request, HttpSession session) {
//        User user = SessionUtils.getUser(session, facade);
//
//        if (user == null) {
//            return new ModelAndView("redirect:/archive/");
//        }

        Long imageId = getImageId(request);
        if (imageId == null || facade.getImageService().findById(imageId) == null) {
            return new ModelAndView("redirect:/archive");
        }

//        if(user.isSuperadmin() || ClientUtils.isImageAdmin(user, facade)) {
        facade.getImageService().unarchiveImage(imageId);
//        }

        return new ModelAndView("redirect:/archive/image/" + imageId);
    }

    private boolean processExport(Long imageId, Image image, Format imageFormat,
                                  ExportImageCommand command, HttpServletResponse response) {

        File tempFile = null;
        try {
            tempFile = facade.getFileService().createTemporaryFile("export");
            File originalFile = facade.getFileService().getImageOriginalFile(imageId, false);

            ImageOp op = new ImageOp(config).input(originalFile);

            Integer width = command.getWidth();
            Integer height = command.getHeight();
            if (width != null || height != null) {
                Resize resize = (width != null && height != null && !command.isKeepAspectRatio() ? Resize.FORCE : Resize.DEFAULT);
                if (ExportImageCommand.SizeUnit.PECENT.equals(command.getSizeUnit())) {
                    resize = Resize.PERCENT;
                }

                op.filter(Filter.LANCZOS);
                op.resize(width, height, resize);
            }

            op.quality(command.getQuality());
            op.outputFormat(imageFormat);

            if (!op.processToFile(tempFile)) {
                return false;
            }

            if (imageFormat == Format.JPEG) {
                File exifTempFile = facade.getFileService().createTemporaryFile("export_exif");

                ExifData data = new ExifData();
                Exif changedExif = image.getExif();
//                Exif changedExif = image.getChangedExif();
                data.setArtist(changedExif.getArtist());
                data.setCopyright(changedExif.getCopyright());
                data.setDescription(changedExif.getDescription());

                boolean res = ExifUtils.writeExifData(tempFile, data, exifTempFile);
                if (res) {
                    tempFile.delete();
                    tempFile = exifTempFile;
                } else {
                    exifTempFile.delete();
                }
            }

            response.setContentType(imageFormat.getMimeType());
            response.setContentLength((int) tempFile.length());

            String contentDisposition =
                    (image.getName() != null && !image.getName().equals("")) ?
                            String.format("attachment; filename=" + image.getName() + ".%s", imageFormat.getExtension()) :
                            String.format("attachment; filename=export_img_%d.%s", imageId, imageFormat.getExtension());

            response.setHeader("Content-Disposition", contentDisposition);

            OutputStream output = null;
            InputStream input = null;
            try {
                input = new FileInputStream(tempFile);
                output = new BufferedOutputStream(response.getOutputStream());

                IOUtils.copy(input, output);
                output.flush();
            } catch (Exception ex) {
                log.warn(ex.getMessage(), ex);
            } finally {
                IOUtils.closeQuietly(output);
                IOUtils.closeQuietly(input);
            }
        } finally {
            if (tempFile != null) {
                tempFile.delete();
            }
        }

        return true;
    }

    private static Collection<Category> getAvailableCategories(Collection<Category> all, Collection<Category> exist) {
        ArrayList<Category> result = new ArrayList<Category>(all);
        result.removeAll(exist);
        return result;
    }

    private static String getCategories(Image image) {
        List<Category> categories = image.getCategories();
        StringBuilder categoryBuilder = new StringBuilder();
        for (int i = 0, len = categories.size(); i < len; i++) {
            Category category = categories.get(i);
            categoryBuilder.append(category.getName());

            if (i < (len - 1)) {
                categoryBuilder.append(", ");
            }
        }

        return categoryBuilder.toString();
    }

    private static String getKeywords(Image image) {
        List<Keyword> keywords = image.getKeywords();
        StringBuilder keywordBuilder = new StringBuilder();
        for (int i = 0, len = keywords.size(); i < len; i++) {
            keywordBuilder.append(keywords.get(i).getName());

            if (i < (len - 1)) {
                keywordBuilder.append(", ");
            }
        }

        return keywordBuilder.toString();
    }
}
