package com.imcode.imcms.addon.imagearchive.controller.web;

import com.imcode.imcms.addon.imagearchive.Config;
import com.imcode.imcms.addon.imagearchive.command.AddImageUploadCommand;
import com.imcode.imcms.addon.imagearchive.command.ChangeImageDataCommand;
import com.imcode.imcms.addon.imagearchive.entity.*;
import com.imcode.imcms.addon.imagearchive.json.UploadResponse;
import com.imcode.imcms.addon.imagearchive.service.Facade;
import com.imcode.imcms.addon.imagearchive.service.ImageService;
import com.imcode.imcms.addon.imagearchive.util.ClientUtils;
import com.imcode.imcms.addon.imagearchive.util.KeywordComparator;
import com.imcode.imcms.addon.imagearchive.util.SessionUtils;
import com.imcode.imcms.addon.imagearchive.util.Utils;
import com.imcode.imcms.addon.imagearchive.validator.ChangeImageDataValidator;
import com.imcode.imcms.addon.imagearchive.validator.ImageUploadValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.validation.ValidationUtils;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.util.*;


@Controller
public class AddImageController {
	private static final Log log = LogFactory.getLog(AddImageController.class);

	@Autowired
	private Facade facade;

	@Autowired
	private Config config;

	@Autowired
	private ImageService imageService;

	@RequestMapping("/add-image")
	public ModelAndView indexHandler(HttpServletRequest request, HttpServletResponse response, HttpSession session) {
		User user = SessionUtils.getUser(request, session, facade);
		if (user == null || user.isDefault()) {
			ClientUtils.redirectToLogin(response, facade);

			return null;
		}

		ModelAndView mav = new ModelAndView("add_image");
		Set<Category> categories = new HashSet<Category>();
		for (Role role : user.getRoles()) {
			categories.addAll(facade.getRoleService().getUsableRoleCategories(role.getId()));
		}

		List<Keyword> keyWords = facade.getKeywordService().getKeywords();
		Collections.sort(keyWords, new KeywordComparator());

        mav.addObject("userName", user.getFirstName() + " " + user.getLastName());
		mav.addObject("upload", new AddImageUploadCommand());
		mav.addObject("keywords", keyWords);
		mav.addObject("categories", categories);

		return mav;
	}

	@RequestMapping(value = "/add-image/multi-upload-form", method = RequestMethod.GET)
	public String multiFileDataFromHandler(HttpServletRequest request,
										   HttpServletResponse response,
										   HttpSession session,
										   ModelAndView mav) {
		User user = SessionUtils.getUser(request, session, facade);
		if (user == null) {
			ClientUtils.redirectToLogin(response, facade);

			return null;
		}

		List<Keyword> keyWords = facade.getKeywordService().getKeywords();
		Collections.sort(keyWords, new KeywordComparator());

		mav.addObject("keywords", keyWords);
		mav.addObject("categories", facade.getCategoryService().getCategories());

		return "fragments/multi_file_data";
	}

	/* Called by uploadify for multi-file upload(images, zips or both).
	 * Stores images and images from zip archives
	 * One request, one file */
	@RequestMapping("/add-image/upload")
	@ResponseBody
	public UploadResponse uploadHandler(
			@ModelAttribute("upload") AddImageUploadCommand command,
			BindingResult result,
			@ModelAttribute("changeData") ChangeImageDataCommand changeData,
			BindingResult dataResult,
			HttpServletRequest request,
			HttpServletResponse response,
			HttpSession session) {
		String contextPath = request.getContextPath();
		User user = SessionUtils.getUser(request, session, facade);
		if (user == null) {
			ClientUtils.redirectToLogin(response, facade);

			return null;
		}

		//Валидатор изображения, создает временный файл с изображением
		ImageUploadValidator validator = new ImageUploadValidator(facade);
		ValidationUtils.invokeValidator(validator, command.getFile(), result);

		if ((command.getFileCount() > 0 || validator.isZipFile()) && changeData.isSubmitted()) {
			//Валидатор информации о картинке
			ChangeImageDataValidator dataValidator = new ChangeImageDataValidator(facade);
			ValidationUtils.invokeValidator(dataValidator, changeData, dataResult);
		}

		UploadResponse uploadResponse = new UploadResponse();
		if (!result.hasErrors() && !dataResult.hasErrors()) {
			try {
				if (validator.isZipFile()) {
					List<Image> images = facade.getImageService().createImagesFromZip(validator.getTempFile());
					if (changeData.isSubmitted()) {
						for (Image image : images) {
							if (image != null) {
								changeData.setImageNm(image.getName());
								changeData.toImage(image);
								facade.getImageService().updateData(image, changeData.getCategoryIds(), changeData.getImageKeywordNames());
							}
						}
					}

				} else {
					Image image = imageService.createImage(validator.getTempFile(),
                            validator.getImageInfo(), validator.getImageName(), Image.STATUS_ACTIVE);

                    if (command.getFileCount() > 1) {
                        // set name as file name only when we have multi-upload
                        changeData.setImageNm(image.getName());
                    }

					if (image == null) {
						result.rejectValue("file", "addImage.invalidImageError");
					} else {
						if (changeData.isSubmitted()) {
							changeData.toImage(image);
							imageService.updateData(image, changeData.getCategoryIds(), changeData.getImageKeywordNames());
						}
					}
				}

				if (command.isRedirToSearch()) {
					uploadResponse.setRedirectOnAllComplete(contextPath + "/archive");
				}

			} catch (Exception ex) {
				log.fatal(ex.getMessage(), ex);
				result.rejectValue("file", "addImage.invalidImageError");
			} finally {
				facade.getFileService().deleteTemporaryFile(validator.getTempFile());
			}
		}

		if (result.hasErrors()) {
			List<String> errors = new ArrayList<String>();
			for (FieldError error : result.getFieldErrors()) {
				errors.add(facade.getMessageSource().getMessage(error.getCode(), error.getArguments(), request.getLocale()));
			}
			uploadResponse.setImageErrors(errors);
		}

		if (dataResult.hasErrors()) {
			Map<String, String> errors = getDataErrors(dataResult, request.getLocale());
			uploadResponse.setDataErrors(errors);
		}
		return uploadResponse;
	}

	private Map<String, String> getDataErrors(BindingResult dataResult, Locale locale) {
		Map<String, String> errors = new HashMap<String, String>();

		for (FieldError error : dataResult.getFieldErrors()) {
			String message = facade.getMessageSource().getMessage(error.getCode(), error.getArguments(), locale);
			errors.put(error.getField(), message);
		}
		return errors;
	}

	@RequestMapping("/add-image/verify-data")
	public void verifyDataHandler(
			@ModelAttribute("changeData") ChangeImageDataCommand changeData,
			BindingResult dataResult,
			HttpServletRequest request,
			HttpServletResponse response) {

		UploadResponse uploadResponse = new UploadResponse();

		//Валидация Данных изображения можно переделать на стандартный валидатор от hibernate.
		if (changeData.isSubmitted()) {
			ChangeImageDataValidator dataValidator = new ChangeImageDataValidator(facade);
			ValidationUtils.invokeValidator(dataValidator, changeData, dataResult);
		}

		if (dataResult.hasErrors()) {
			Map<String, String> errors = getDataErrors(dataResult, request.getLocale());
			uploadResponse.setDataErrors(errors);
		}

		Utils.writeJSON(uploadResponse, response);
	}
}
