package com.imcode.imcms.addon.imagearchive.controller.web;

import com.imcode.imcms.addon.imagearchive.SessionConstants;
import com.imcode.imcms.addon.imagearchive.command.SearchImageCommand;
import com.imcode.imcms.addon.imagearchive.entity.*;
import com.imcode.imcms.addon.imagearchive.service.Facade;
import com.imcode.imcms.addon.imagearchive.util.KeywordComparator;
import com.imcode.imcms.addon.imagearchive.util.Pagination;
import com.imcode.imcms.addon.imagearchive.util.SessionUtils;
import com.imcode.imcms.addon.imagearchive.util.Utils;
import com.imcode.imcms.addon.imagearchive.validator.SearchImageValidator;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.validation.ValidationUtils;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import java.io.IOException;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

@Controller
public class SearchImageController {
	private static final Log log = LogFactory.getLog(SearchImageController.class);

	private static final Pattern LOCAL_PAGE_PATTERN = Pattern.compile("/client/archive/page/(\\d+)?");
    private static final Pattern PAGE_PATTERN = Pattern.compile("/archive/page/(\\d+)?");

    private static final String PAGINATION_KEY = Utils.makeKey(SearchImageController.class, "pagination");
	private static final String COMMAND_KEY = Utils.makeKey(SearchImageController.class, "command");

	@Autowired
	private Facade facade;

	private static Pagination getPagination(HttpSession session) {
		Pagination pag = (Pagination) session.getAttribute(PAGINATION_KEY);
		if (pag == null) {
			pag = new Pagination(SearchImageCommand.DEFAULT_PAGE_SIZE);
			session.setAttribute(PAGINATION_KEY, pag);
		}

		return pag;
	}

	private static int getPage(HttpServletRequest request) {
        final String requestURI = request.getRequestURI();

        Pattern pattern = (requestURI.startsWith("/archive"))
                ? PAGE_PATTERN
                : LOCAL_PAGE_PATTERN;

        Matcher matcher = pattern.matcher(requestURI);

		int page = 0;
		if (matcher.find()) {
			try {
				page = Integer.parseInt(matcher.group(1), 10);
				page = Math.max(page - 1, 0);
			} catch (NumberFormatException ignore) {
			}
		}

		return page;
	}

	@RequestMapping({"/archive", "/"})
	public ModelAndView indexHandler(
			@ModelAttribute("search") SearchImageCommand command,
			BindingResult result,
			@RequestParam(required = false) String returnTo,
			HttpServletRequest request,
			HttpSession session) throws IOException {

		SearchImageCommand temporaryCommand = new SearchImageCommand();
		temporaryCommand.copyFrom(command);
		returnTo = StringUtils.trimToNull(returnTo);

		if (returnTo != null) {
			session.setAttribute(SessionConstants.IMCMS_RETURN_URL, returnTo);
		}

		User user = SessionUtils.getUser(request, session, facade);

		if (user == null) {
			return null;
		}

		ModelAndView mav = new ModelAndView("search_image");
		Pagination pag = getPagination(session);

		if (request.getParameter("show") == null) {
			SearchImageCommand cmd = (SearchImageCommand) session.getAttribute(COMMAND_KEY);
			if (cmd != null) {
				command.copyFrom(cmd);
			}
		} else {
			session.setAttribute(COMMAND_KEY, command);
			pag.setCurrentPage(0);
		}

		if (command.isClear()) {
			command.copyFrom(new SearchImageCommand());
		}

		mav.addObject("search", command);

		Set<Category> categories = new HashSet<Category>();
		for (Role role : user.getRoles()) {
			categories.addAll(facade.getRoleService().getUsableRoleCategories(role.getId()));
		}

		List<Keyword> keywords = facade.getKeywordService().getKeywords();
		Collections.sort(keywords, new KeywordComparator());

        List<String> artists = new ArrayList<String>(facade.getExifService().getAllArtists());
        Collections.sort(artists);

		mav.addObject("categories", categories);
		mav.addObject("keywords", keywords);
		mav.addObject("artists", artists);

		SearchImageValidator validator = new SearchImageValidator(categories);
		ValidationUtils.invokeValidator(validator, temporaryCommand, result);

		if (result.hasErrors()) {
			return mav;
		}

		session.setAttribute(COMMAND_KEY, command);

        int allImages = facade.getImageService().searchImages(temporaryCommand).size();
        pag.setPageSize(command.getResultsPerPage());
        pag.update(allImages);
		List<Image> images = facade.getImageService().searchImages(pag, temporaryCommand);
		mav.addObject("imageCount", images.size());
		mav.addObject("images", images);
		mav.addObject("pag", pag);

		return mav;
	}

	@RequestMapping("/page/*")
	public ModelAndView pageHandler(HttpServletRequest request,
                                    @ModelAttribute("search") SearchImageCommand command,
                                    BindingResult result,
                                    HttpSession session) {

	    User user = SessionUtils.getUser(request, session, facade);
        command = (SearchImageCommand) session.getAttribute(COMMAND_KEY);

        if (command == null) {
            return new ModelAndView("redirect:/archive/");
        }

		Pagination pag = getPagination(session);
		pag.setCurrentPage(getPage(request));

		ModelAndView mav = new ModelAndView("search_image");
		mav.addObject("search", command);

		List<Keyword> keywords = facade.getKeywordService().getKeywords();
		Collections.sort(keywords, new KeywordComparator());

        List<String> artists = new ArrayList<String>(facade.getExifService().getAllArtists());
        Collections.sort(artists);

        mav.addObject("keywords", keywords);
		mav.addObject("artists", artists);
		Set<Category> categories = new HashSet<Category>();

        for (Role role : user.getRoles()) {
			categories.addAll(facade.getRoleService().getUsableRoleCategories(role.getId()));
		}

		mav.addObject("categories", categories);

        SearchImageCommand temporaryCommand = new SearchImageCommand();
        temporaryCommand.copyFrom(command);

        SearchImageValidator validator = new SearchImageValidator(categories);
        ValidationUtils.invokeValidator(validator, temporaryCommand, result);

        if (result.hasErrors()) {
            return mav;
        }

		int imageCount = facade.getImageService().searchImages(temporaryCommand).size();
        mav.addObject("imageCount", imageCount);

		pag.update(imageCount);
		List<Image> images = facade.getImageService().searchImages(pag, temporaryCommand);

		mav.addObject("images", images);
		mav.addObject("pag", pag);

		return mav;
	}
}
