package com.imcode.imcms.addon.imagearchive.controller.web;

import com.imcode.imcms.addon.imagearchive.command.*;
import com.imcode.imcms.addon.imagearchive.dto.LibraryRolesDto;
import com.imcode.imcms.addon.imagearchive.entity.*;
import com.imcode.imcms.addon.imagearchive.repository.CategoryRolesRepository;
import com.imcode.imcms.addon.imagearchive.service.Facade;
import com.imcode.imcms.addon.imagearchive.service.exception.CategoryExistsException;
import com.imcode.imcms.addon.imagearchive.service.exception.KeywordExistsException;
import com.imcode.imcms.addon.imagearchive.service.impl.ClientCategoryService;
import com.imcode.imcms.addon.imagearchive.util.ClientUtils;
import com.imcode.imcms.addon.imagearchive.util.SessionUtils;
import com.imcode.imcms.addon.imagearchive.util.Utils;
import com.imcode.imcms.addon.imagearchive.validator.CreateCategoryValidator;
import com.imcode.imcms.addon.imagearchive.validator.CreateKeywordValidator;
import com.imcode.imcms.addon.imagearchive.validator.EditCategoryValidator;
import com.imcode.imcms.addon.imagearchive.validator.EditKeywordValidator;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.Predicate;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.validation.ValidationUtils;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.File;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

@Controller
public class PreferencesController {
    private static final Log log = LogFactory.getLog(PreferencesController.class);

    private static final String ROLE_KEY = Utils.makeKey(PreferencesController.class, "role");
    private static final String LIBRARY_KEY = Utils.makeKey(PreferencesController.class, "library");

    @Autowired
    private Facade facade;

    @Autowired
    private ClientCategoryService clientCategoryService;

    @Autowired
    private CategoryRolesRepository categoryRolesRepository;

    private static Role getRole(HttpSession session, List<Role> roles) {
        Role role = (Role) session.getAttribute(ROLE_KEY);
        if (role == null && roles != null) {
            role = roles.get(0);
            session.setAttribute(ROLE_KEY, role);
        }

        return role;

    }

    private static Library getLibrary(HttpSession session, List<Library> libraries) {
        Library library = (Library) session.getAttribute(LIBRARY_KEY);

        if (library != null && libraries != null) {
            boolean exists = false;
            for (Library lib : libraries) {
                if (lib.getId() == library.getId()) {
                    exists = true;
                    break;
                }
            }

            if (!exists) {
                library = null;
                session.removeAttribute(LIBRARY_KEY);
            }

        }

        if (library == null && libraries != null && !libraries.isEmpty()) {
            library = libraries.get(0);
            session.setAttribute(LIBRARY_KEY, library);
        }

        return library;
    }

    @RequestMapping("/preferences")
    public String indexHandler(
            @ModelAttribute PreferencesActionCommand actionCommand,

            @ModelAttribute("createCategory") CreateCategoryCommand createCategoryCommand,
            BindingResult createCategoryResult,

            @ModelAttribute("editCategory") EditCategoryCommand editCategoryCommand,
            BindingResult editCategoryResult,

            @ModelAttribute("createKeyword") CreateKeywordCommand createKeywordCommand,
            BindingResult createKeywordResult,

            @ModelAttribute("editKeyword") EditKeywordCommand editKeywordCommand,
            BindingResult editKeywordResult,

            @ModelAttribute("saveLibraryRoles") SaveLibraryRolesCommand librariesCommand,
            BindingResult librariesResult,

            @ModelAttribute("saveCategories") SaveRoleCategoriesCommand roleCategoriesCommand,
            HttpServletRequest request,
            HttpServletResponse response,
            HttpSession session,
            Map<String, Object> model) {

        User user = SessionUtils.getUser(request, session, facade);
        if (user == null) {
            ClientUtils.redirectToLogin(response, facade);

            return null;
        } else if (!ClientUtils.canAccessPreferences(user, facade)) {
            return "redirect:/archive";
        }

        facade.getLibraryService().syncLibraryFolders();

        List<Role> roles = facade.getRoleService().getAllRoles();
        Role role = getRole(session, roles);
        Role previousRole = (Role) session.getAttribute("previousRole");
        if (previousRole != null && !role.equals(previousRole)) {
            model.put("editingRoles", true);
        }
        session.setAttribute("previousRole", role);
        model.put("currentRole", role);

        List<Library> libraries = facade.getLibraryService().findLibraries();
        final List<File> firstLevelLibraries = facade.getFileService().listFirstLevelLibraryFolders();
        CollectionUtils.filter(libraries, new Predicate<Library>() {
            public boolean evaluate(Library lib) {
                return firstLevelLibraries.contains(new File(lib.getFilepath(), lib.getFolderNm()));
            }
        });

        Library library = getLibrary(session, libraries);
//        Library previousLibrary = (Library) session.getAttribute("previousLibrary");
//        if (previousLibrary != null && !library.equals(previousLibrary)) {
//            model.put("editingLibraries", true);
//        }
        session.setAttribute("previousLibrary", library);
        model.put("currentLibrary", library);

        if (actionCommand.isCreateCategory()) {
            processCreateCategory(createCategoryCommand, createCategoryResult);
            model.put("editingCategories", true);

        } else if (actionCommand.isEditCategory()) {
            processEditCategory(editCategoryCommand);
            model.put("editingCategories", true);

        } else if (actionCommand.isSaveCategory()) {
            processSaveCategory(editCategoryCommand, editCategoryResult);
            model.put("editingCategories", true);

        } else if (actionCommand.isRemoveCategory()) {
            processRemoveCategory(editCategoryCommand);
            model.put("editingCategories", true);

        } else if (actionCommand.isCreateKeyword()) {
            processCreateKeyword(createKeywordCommand, createKeywordResult);
            model.put("editingKeywords", true);

        } else if (actionCommand.isSaveKeyword()) {
            processSaveKeyword(editKeywordCommand, editKeywordResult);
            model.put("editingKeywords", true);

        } else if (actionCommand.isRemoveKeyword()) {
            processRemoveKeyword(editKeywordCommand);
            model.put("editingKeywords", true);

        } else if (actionCommand.isSaveLibraryRoles() && library != null && librariesCommand.getLibraryRoles() != null) {
            processSaveLibraryRoles(librariesCommand, librariesResult, model);
            model.put("editingLibraries", true);

        } else if (actionCommand.isSaveRoleCategories()) {
            processSaveRoleCategories(roleCategoriesCommand, model);
            model.put("editingRoles", true);

        }

        if (!actionCommand.isSaveLibraryRoles() && library != null) {
            librariesCommand.setLibraryNm(library.getLibraryNm());
        }

//        List<Role> availableLibraryRoles = Collections.emptyList();
        List<LibraryRolesDto> libraryRoles = Collections.emptyList();

        if (library != null) {
//            availableLibraryRoles = facade.getLibraryService().getAllRoles();

            libraryRoles = facade.getLibraryService().findLibraryRoles(library.getId());
        }

//        model.put("availableLibraryRoles", availableLibraryRoles);
        model.put("availableLibraryRoles", null);
        model.put("libraryRoles", libraryRoles);

        List<CategoryRoles> categoryRoles = facade.getRoleService().getCategoryRolesByRole(role);
        model.put("categoryRoles", categoryRoles);

        Collection<CategoryType> categoryTypes = facade.getCategoryTypeService().getCategoryTypes();

        // get catTypes only associated with Image Archive
        CollectionUtils.filter(categoryTypes, new Predicate<CategoryType>() {
            public boolean evaluate(CategoryType categoryType) {
                return categoryType.isImageArchive();
            }
        });

        // get all categories id from types
        Collection<Long> typeCategoriesId = facade.getCategoryTypeService().getCategoryIdsOfTypes(categoryTypes);
        Collection<Category> typeCategories = facade.getCategoryService().getCategories(typeCategoriesId);

        model.put("categoryTypes", categoryTypes);
        model.put("categories", typeCategories);

        model.put("keywords", facade.getKeywordService().getKeywords());

        model.put("roles", roles);

        model.put("libraries", libraries);
//        model.put("freeCategories", facade.getRoleService().findFreeCategories(role.getId()));
//        model.put("roleCategories", facade.getRoleService().findRoleCategories(role.getId()));
//        model.put("allCategories", facade.getRoleService().findAllCategories());

        model.put("freeCategories", typeCategories);
        model.put("roleCategories", typeCategories);
        model.put("allCategories", typeCategories);

        return "preferences";
    }

    private void processCreateCategory(CreateCategoryCommand command, BindingResult result) {
        CreateCategoryValidator validator = new CreateCategoryValidator();
        ValidationUtils.invokeValidator(validator, command, result);

        if (!result.hasErrors()) {
            String categoryName = command.getCreateCategoryName();
            Integer categoryTypeId = command.getCreateCategoryType();

            // this done because for client and server we could have different DB, so next code synchronize categories
            try {
                Long catId = facade.getCategoryService().createCategory(categoryName, categoryTypeId).getId();
                Long clientCatId = (clientCategoryService.getCategory(categoryName) == null)
                        ? clientCategoryService.createCategory(categoryName, categoryTypeId.longValue()).getId()
                        : clientCategoryService.getCategory(categoryName).getId();

                while (!catId.equals(clientCatId)) {
                    if (catId < clientCatId) {
                        facade.getCategoryService().deleteCategory(catId);
                        catId = facade.getCategoryService().createCategory(categoryName, categoryTypeId).getId();

                    } else if (catId > clientCatId) {
                        final List<CategoryRoles> categoryRoles = categoryRolesRepository.findByCategoryId(clientCatId.intValue());

                        if (!categoryRoles.isEmpty()) {
                            categoryRolesRepository.delete(categoryRoles);
                        }

                        clientCategoryService.deleteCategory(clientCatId);
                        clientCatId = clientCategoryService.createCategory(categoryName, categoryTypeId.longValue()).getId();

                        if (!categoryRoles.isEmpty()) {
                            for (CategoryRoles categoryRole : categoryRoles) {
                                categoryRole.setCategoryId(clientCatId.intValue());
                            }

                            categoryRolesRepository.save(categoryRoles);
                        }
                    }
                }

                command.setCreateCategoryName("");
                command.setCreateCategoryType(0);
            } catch (CategoryExistsException ex) {
                result.rejectValue("createCategoryName", "preferences.categoryExistsError");
            }
        }
    }

    private void processEditCategory(EditCategoryCommand command) {
        Category category = facade.getCategoryService().getCategory(Long.valueOf(command.getEditCategoryId()));
        if (category != null) {
            command.setShowEditCategory(true);
            command.setEditCategoryName(category.getName());
        }
    }

    private void processSaveCategory(EditCategoryCommand command, BindingResult result) {
        EditCategoryValidator validator = new EditCategoryValidator();
        ValidationUtils.invokeValidator(validator, command, result);

        if (!result.hasErrors()) {
            long categoryId = command.getEditCategoryId();
            String newName = command.getEditCategoryName();

            try {
                facade.getCategoryService().updateCategory(categoryId, newName);
                clientCategoryService.updateCategory(categoryId, newName);

            } catch (CategoryExistsException ex) {
                result.rejectValue("editCategoryName", "preferences.categoryExistsError");
            }
        }
    }

    private void processRemoveCategory(EditCategoryCommand command) {
        final int categoryId = command.getEditCategoryId();
        final List<CategoryRoles> categoryRoles = categoryRolesRepository.findByCategoryId(categoryId);

        facade.getCategoryService().deleteCategory((long) categoryId);
        categoryRolesRepository.delete(categoryRoles);
        clientCategoryService.deleteCategory((long) categoryId);

        command.setEditCategoryName("");
        command.setShowEditCategory(false);
    }

    private void processCreateKeyword(CreateKeywordCommand command, BindingResult result) {
        CreateKeywordValidator validator = new CreateKeywordValidator();
        ValidationUtils.invokeValidator(validator, command, result);

        if (!result.hasErrors()) {
            String keywordName = command.getCreateKeywordName();

            try {
                facade.getKeywordService().createKeyword(keywordName);

                command.setCreateKeywordName("");
            } catch (KeywordExistsException ex) {
                result.rejectValue("createKeywordName", "preferences.keywordExistsError");
            }
        }
    }

    private void processSaveKeyword(EditKeywordCommand command, BindingResult result) {
        EditKeywordValidator validator = new EditKeywordValidator();
        ValidationUtils.invokeValidator(validator, command, result);

        if (!result.hasErrors()) {
            long keywordId = command.getEditKeywordId();
            String KeywordName = command.getEditKeywordName();

            try {
                facade.getKeywordService().updateKeyword(keywordId, KeywordName);

            } catch (KeywordExistsException ex) {
                result.rejectValue("editKeywordName", "preferences.keywordExistsError");
            }
        }
    }

    private void processRemoveKeyword(EditKeywordCommand command) {
        facade.getKeywordService().deleteKeyword(command.getEditKeywordId());

        command.setEditKeywordName("");
    }

    private void processSaveLibraryRoles(SaveLibraryRolesCommand command, BindingResult result, Map<String, Object> model) {
//        SaveLibraryRolesValidator validator = new SaveLibraryRolesValidator();
//        ValidationUtils.invokeValidator(validator, command, result);
//
//        Library library = (Library) model.get("currentLibrary");
//
//        if (!result.hasErrors()) {
//            try {
//                facade.getLibraryService().updateLibraryRoles(library.getId(), command.getLibraryNm(), command.getLibraryRoles());
//
//                library.setLibraryNm(command.getLibraryNm());
//            } catch (Exception ex) {
//                log.warn(ex.getMessage(), ex);
//            }
//        }
    }

    private void processSaveRoleCategories(SaveRoleCategoriesCommand command, Map<String, Object> model) {
        try {
            Role role = (Role) model.get("currentRole");

            facade.getRoleService().assignCategoryRoles(role, command.getAssignedCategoryIds());
        } catch (Exception ex) {
            log.fatal(ex.getMessage(), ex);
        }
    }

    @RequestMapping("/preferences/role")
    public String changeCurrentRoleHandler(
            @RequestParam(required = false) Integer id,
            HttpServletRequest request,
            HttpServletResponse response,
            HttpSession session) {
        User user = SessionUtils.getUser(request, session, facade);
        if (user == null) {
            ClientUtils.redirectToLogin(response, facade);

            return null;
        } else if (!ClientUtils.canAccessPreferences(user, facade)) {
            return "redirect:/archive";
        }

        Role role;
        if (id != null && (role = facade.getRoleService().getRoleById(id)) != null) {
            session.setAttribute(ROLE_KEY, role);
        }

        return "redirect:/archive/preferences";
    }

    @RequestMapping("/preferences/library")
    public String changeCurrentLibraryHander(
            @RequestParam(required = false) Long id,
            HttpServletResponse response,
            HttpSession session) {
//        User user = SessionUtils.getUser(session, facade);
//        if (user == null) {
//            ClientUtils.redirectToLogin(response, facade);
//
//            return null;
//        } else if (!ClientUtils.canAccessPreferences(user, facade)) {
//            return "redirect:/archive";
//        }

        Library library;
        if (id != null && (library = facade.getLibraryService().findLibraryById(id)) != null) {
            session.setAttribute(LIBRARY_KEY, library);
        }

        return "redirect:/archive/preferences";
    }
}
