package com.imcode.imcms.addon.imagearchive.controller.web;

import com.imcode.imcms.addon.imagearchive.SessionConstants;
import com.imcode.imcms.addon.imagearchive.command.SearchImageCommand;
import com.imcode.imcms.addon.imagearchive.entity.Category;
import com.imcode.imcms.addon.imagearchive.entity.Image;
import com.imcode.imcms.addon.imagearchive.entity.Role;
import com.imcode.imcms.addon.imagearchive.entity.User;
import com.imcode.imcms.addon.imagearchive.service.Facade;
import com.imcode.imcms.addon.imagearchive.util.Pagination;
import com.imcode.imcms.addon.imagearchive.util.SessionUtils;
import com.imcode.imcms.addon.imagearchive.util.Utils;
import com.imcode.imcms.addon.imagearchive.validator.SearchImageValidator;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.validation.ValidationUtils;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

@Controller
public class SearchImageController {
	private static final Log log = LogFactory.getLog(SearchImageController.class);
	private static final Pattern PAGE_PATTERN = Pattern.compile("/archive/page/(\\d+)/?");
	private static final String PAGINATION_KEY = Utils.makeKey(SearchImageController.class, "pagination");
	private static final String COMMAND_KEY = Utils.makeKey(SearchImageController.class, "command");

	@Autowired
	private Facade facade;

	@RequestMapping({"/archive", "/"})
	public ModelAndView indexHandler(
			@ModelAttribute("search") SearchImageCommand command,
			BindingResult result,
			@RequestParam(required = false) String returnTo,
			HttpServletRequest request,
			HttpSession session) throws IOException {
		SearchImageCommand temporaryCommand = new SearchImageCommand();
		temporaryCommand.copyFrom(command);
		returnTo = StringUtils.trimToNull(returnTo);
		if (returnTo != null) {
			session.setAttribute(SessionConstants.IMCMS_RETURN_URL, returnTo);
		}

		User user = SessionUtils.getUser(request, session, facade);

		if (user == null) {
			return null;
		}

		ModelAndView mav = new ModelAndView("search_image");
		Pagination pag = getPagination(session);

		if (request.getParameter("show") == null) {
			SearchImageCommand cmd = (SearchImageCommand) session.getAttribute(COMMAND_KEY);
			if (cmd != null) {
				command.copyFrom(cmd);
			}
		} else {
			session.setAttribute(COMMAND_KEY, command);
			pag.setCurrentPage(0);
		}

		if (command.isClear()) {
			command.copyFrom(new SearchImageCommand());
		}

		mav.addObject("search", command);

		Set<Category> categories = new HashSet<>();
		for (Role role : user.getRoles()) {
			categories.addAll(facade.getRoleService().getUsableRoleCategories(role.getId()));
		}

		mav.addObject("categories", categories);
		mav.addObject("keywords", facade.getKeywordService().getKeywords());
		mav.addObject("artists", "");

		SearchImageValidator validator = new SearchImageValidator(categories);
		ValidationUtils.invokeValidator(validator, temporaryCommand, result);

		if (result.hasErrors()) {
			return mav;
		}

		session.setAttribute(COMMAND_KEY, command);

		pag.setPageSize(command.getResultsPerPage());
		List<Image> images = facade.getImageService().searchImages(pag, temporaryCommand);
		mav.addObject("imageCount", images.size());
		mav.addObject("images", images);
		mav.addObject("pag", pag);

		return mav;
	}

	@RequestMapping("/page/*")
	public ModelAndView pageHandler(HttpServletRequest request,
									HttpServletResponse response,
									HttpSession session) {
		User user = SessionUtils.getUser(request,
				session, facade);

		SearchImageCommand command = (SearchImageCommand) session.getAttribute(COMMAND_KEY);
		if (command == null) {
			return new ModelAndView("redirect:/archive/");
		}

		Pagination pag = getPagination(session);
		pag.setCurrentPage(getPage(request));

		ModelAndView mav = new ModelAndView("search_image");
		mav.addObject("search", command);
		mav.addObject("pag", pag);

		mav.addObject("keywords", facade.getKeywordService().getKeywords());
		mav.addObject("artists", "");
		Set<Category> categories = new HashSet<>();
		for (Role role : user.getRoles()) {
			categories.addAll(facade.getRoleService().getUsableRoleCategories(role.getId()));
		}
		mav.addObject("categories", categories);

		List<Long> categoryIds = new ArrayList<>(categories.size());
		categoryIds.addAll(categories.stream().map(Category::getId).collect(Collectors.toList()));

		Long imageCount = facade.getImageService().searchImagesCount(categoryIds);
		mav.addObject("imageCount", imageCount);

		pag.update(imageCount.intValue());
		List<Image> images = facade.getImageService().searchImages(pag, command);
		mav.addObject("images", images);

		return mav;
	}

	private static Pagination getPagination(HttpSession session) {
		Pagination pag = (Pagination) session.getAttribute(PAGINATION_KEY);
		if (pag == null) {
			pag = new Pagination(SearchImageCommand.DEFAULT_PAGE_SIZE);
			session.setAttribute(PAGINATION_KEY, pag);
		}

		return pag;
	}

	private static int getPage(HttpServletRequest request) {
		Matcher matcher = PAGE_PATTERN.matcher(request.getRequestURI());

		int page = 0;
		if (matcher.find()) {
			try {
				page = Integer.parseInt(matcher.group(1), 10);
				page = Math.max(page - 1, 0);
			} catch (NumberFormatException ex) {
			}
		}

		return page;
	}

}
