package com.imcode.imcms.addon.imagearchive.controller.web;

import com.imcode.imcms.addon.imagearchive.command.AddImageUploadCommand;
import com.imcode.imcms.addon.imagearchive.command.ChangeImageDataCommand;
import com.imcode.imcms.addon.imagearchive.entity.Category;
import com.imcode.imcms.addon.imagearchive.entity.Image;
import com.imcode.imcms.addon.imagearchive.entity.Role;
import com.imcode.imcms.addon.imagearchive.entity.User;
import com.imcode.imcms.addon.imagearchive.json.UploadResponse;
import com.imcode.imcms.addon.imagearchive.service.Facade;
import com.imcode.imcms.addon.imagearchive.service.ImageService;
import com.imcode.imcms.addon.imagearchive.util.SessionUtils;
import com.imcode.imcms.addon.imagearchive.util.Utils;
import com.imcode.imcms.addon.imagearchive.validator.ChangeImageDataValidator;
import com.imcode.imcms.addon.imagearchive.validator.ImageUploadValidator;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.validation.ValidationUtils;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.util.*;
import java.util.stream.Collectors;


@Controller
public class AddImageController {
	private static final Log log = LogFactory.getLog(AddImageController.class);

	@Autowired
	private Facade facade;

	@Autowired
	private ImageService imageService;

	@RequestMapping("/add-image")
	public ModelAndView indexHandler(HttpServletRequest request, HttpServletResponse response, HttpSession session) {
		User user = SessionUtils.getUser(request, session, facade);
		if (user == null) {
			Utils.redirectToLogin(response, facade);

			return null;
		}

		ModelAndView mav = new ModelAndView("add_image");
		Set<Category> categories = new HashSet<>();
		for (Role role : user.getRoles()) {
			categories.addAll(facade.getRoleService().getUsableRoleCategories(role.getId()));
		}
		mav.addObject("upload", new AddImageUploadCommand());
		mav.addObject("keywords", facade.getKeywordService().getKeywords());
		mav.addObject("categories", categories);

		return mav;
	}

	@RequestMapping(value = "/add-image/multi-upload-form", method = RequestMethod.GET)
	public String multiFileDataFromHandler(HttpServletRequest request,
										   HttpServletResponse response,
										   HttpSession session,
										   ModelAndView mav) {
		User user = SessionUtils.getUser(request, session, facade);
		if (user == null) {
			Utils.redirectToLogin(response, facade);

			return null;
		}

		mav.addObject("keywords", facade.getKeywordService().getKeywords());
		mav.addObject("categories", facade.getCategoryService().getCategories());

		return "fragments/multi_file_data";
	}

	/* Called by uploadify for multi-file upload(images, zips or both).
	 * Stores images and images from zip archives
	 * One request, one file */
	@RequestMapping("/add-image/upload")
	@ResponseBody
	public UploadResponse uploadHandler(
			@ModelAttribute("upload") AddImageUploadCommand command,
			BindingResult result,
			@ModelAttribute("changeData") ChangeImageDataCommand changeData,
			BindingResult dataResult,
			HttpServletRequest request,
			HttpServletResponse response,
			HttpSession session) {
		String contextPath = request.getContextPath();
		User user = SessionUtils.getUser(request, session, facade);
		if (user == null) {
			Utils.redirectToLogin(response, facade);

			return null;
		}

		//Валидатор изображения, создает временный файл с изображением
		ImageUploadValidator validator = new ImageUploadValidator(facade);
		ValidationUtils.invokeValidator(validator, command.getFile(), result);

		if ((command.getFileCount() > 0 || validator.isZipFile()) && changeData.isSubmitted()) {
			//Валидатор информации о картинке
			ChangeImageDataValidator dataValidator = new ChangeImageDataValidator(facade);
			ValidationUtils.invokeValidator(dataValidator, changeData, dataResult);
		}

		UploadResponse uploadResponse = new UploadResponse();
		if (!result.hasErrors() && !dataResult.hasErrors()) {
			try {
				if (validator.isZipFile()) {
					List<Image> images = facade.getImageService().createImagesFromZip(validator.getTempFile());
					if (changeData.isSubmitted()) {
						for (Image image : images) {
							if (image != null) {
								changeData.setImageNm(image.getName());
								changeData.toImage(image);
								facade.getImageService().updateData(image, changeData.getCategoryIds(), changeData.getImageKeywordNames());
							}
						}
					}

				} else {
					Image image;
					if (command.getFileCount() > 1) {
						image = facade.getImageService().createImage(validator.getTempFile(),
								validator.getImageInfo(), validator.getImageName(), Image.STATUS_ACTIVE);
					} else {
						image = imageService.createImage(validator.getTempFile(),
								validator.getImageInfo(), validator.getImageName(), Image.STATUS_ACTIVE);

						if (changeData.isSubmitted()) {
							if (image != null) {
								changeData.setImageNm(image.getName());
								changeData.toImage(image);
								imageService.updateData(image, changeData.getCategoryIds(), changeData.getImageKeywordNames());
							}
						}
					}

					if (image == null) {
						result.rejectValue("file", "addImage.invalidImageError");
					} else {
						if (changeData.isSubmitted()) {
							changeData.setImageNm(image.getName());
							changeData.toImage(image);
							imageService.updateData(image, changeData.getCategoryIds(), changeData.getImageKeywordNames());
						}
					}
				}

				if (command.isRedirToSearch()) {
					uploadResponse.setRedirectOnAllComplete(contextPath + "/archive");
				}

			} catch (Exception ex) {
				log.fatal(ex.getMessage(), ex);
				result.rejectValue("file", "addImage.invalidImageError");
			} finally {
				facade.getFileService().deleteTemporaryFile(validator.getTempFile());
			}
		}

		if (result.hasErrors()) {
			uploadResponse.setImageErrors(getErrorsList(result, request.getLocale()));
		}

		if (dataResult.hasErrors()) {
			uploadResponse.setDataErrors(getErrorsMap(dataResult, request.getLocale()));
		}
		return uploadResponse;
	}

	private Map<String, String> getErrorsMap(BindingResult result, Locale locale) {
		Map<String, String> errors = new HashMap<>();
		for (FieldError error : result.getFieldErrors()) {
			errors.put(error.getField(), facade.getCommonService().getMessage(error.getCode(), locale, error.getArguments()));
		}
		return errors;
	}

	private List<String> getErrorsList(BindingResult result, Locale locale) {
		return result.getFieldErrors().stream()
				.map(error -> facade.getCommonService().getMessage(error.getCode(), locale, error.getArguments()))
				.collect(Collectors.toList());
	}

	@RequestMapping("/add-image/verify-data")
	public void verifyDataHandler(
			@ModelAttribute("changeData") ChangeImageDataCommand changeData,
			BindingResult dataResult,
			HttpServletRequest request,
			HttpServletResponse response,
			HttpSession session) {

		UploadResponse uploadResponse = new UploadResponse();

		//Валидация Данных изображения можно переделать на стандартный валидатор от hibernate.
		if (changeData.isSubmitted()) {
			ChangeImageDataValidator dataValidator = new ChangeImageDataValidator(facade);
			ValidationUtils.invokeValidator(dataValidator, changeData, dataResult);
		}

		if (dataResult.hasErrors()) {
			uploadResponse.setDataErrors(getErrorsMap(dataResult, request.getLocale()));
		}

		Utils.writeJSON(uploadResponse, response);
	}
}
