/*
 * Decompiled with CFR 0.152.
 */
package imcode.server.document.index;

import imcode.server.document.DocumentDomainObject;
import imcode.server.document.index.DirectoryIndex;
import imcode.server.document.index.DocumentIndex;
import imcode.server.document.index.IndexBuildingThread;
import imcode.server.document.index.IndexException;
import imcode.server.user.UserDomainObject;
import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Timer;
import java.util.TimerTask;
import org.apache.log4j.Logger;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Query;

public class AutorebuildingDirectoryIndex
implements DocumentIndex {
    private static final Logger log = Logger.getLogger((String)AutorebuildingDirectoryIndex.class.getName());
    private final DirectoryIndex index;
    private final File indexDirectory;
    private final int indexingSchedulePeriodInMilliseconds;
    private IndexBuildingThread indexBuildingThread;

    public AutorebuildingDirectoryIndex(File indexDirectory, int indexingSchedulePeriodInMinutes) {
        this.indexDirectory = indexDirectory;
        this.indexingSchedulePeriodInMilliseconds = indexingSchedulePeriodInMinutes * 60000;
        this.index = new DirectoryIndex(indexDirectory);
        Timer scheduledIndexBuildingTimer = new Timer(true);
        long scheduledIndexDelay = 0L;
        if (IndexReader.indexExists((File)indexDirectory)) {
            try {
                long indexModifiedTime = IndexReader.lastModified((File)indexDirectory);
                long time = System.currentTimeMillis();
                long nextTime = indexModifiedTime + (long)this.indexingSchedulePeriodInMilliseconds;
                if (nextTime > time) {
                    log.info((Object)("First indexing scheduled at " + this.formatDatetime(new Date(nextTime))));
                    scheduledIndexDelay = nextTime - time;
                }
            }
            catch (IOException e) {
                log.warn((Object)"Failed to get last modified time of index.", (Throwable)e);
            }
        }
        scheduledIndexBuildingTimer.scheduleAtFixedRate((TimerTask)new ScheduledIndexingTimerTask(), scheduledIndexDelay, (long)this.indexingSchedulePeriodInMilliseconds);
    }

    private String formatDatetime(Date nextExecutionTime) {
        return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(nextExecutionTime);
    }

    public void indexDocument(DocumentDomainObject document) {
        log.debug((Object)"Adding document.");
        if (null != this.indexBuildingThread) {
            this.indexBuildingThread.addDocument(document);
        }
        try {
            this.index.indexDocument(document);
        }
        catch (IndexException e) {
            this.rebuildBecauseOfError("Failed to add document " + document.getId() + " to index.", e);
        }
    }

    public void removeDocument(DocumentDomainObject document) {
        log.debug((Object)"Removing document.");
        if (null != this.indexBuildingThread) {
            this.indexBuildingThread.removeDocument(document);
        }
        try {
            this.index.removeDocument(document);
        }
        catch (IndexException e) {
            this.rebuildBecauseOfError("Failed to remove document " + document.getId() + " from index.", e);
        }
    }

    public DocumentDomainObject[] search(Query query, UserDomainObject searchingUser) throws IndexException {
        try {
            DocumentDomainObject[] documents = this.index.search(query, searchingUser);
            if (this.index.isInconsistent()) {
                this.rebuildBecauseOfError("Index is inconsistent.", null);
            }
            return documents;
        }
        catch (IndexException ex) {
            this.rebuildBecauseOfError("Search failed.", ex);
            return new DocumentDomainObject[0];
        }
    }

    private void rebuildBecauseOfError(String message, IndexException ex) {
        log.error((Object)(message + " Rebuilding index..."), (Throwable)ex);
        this.rebuild();
    }

    public void rebuild() {
        this.rebuildInBackground();
    }

    private void rebuildInBackground() {
        if (null == this.indexBuildingThread || !this.indexBuildingThread.isAlive()) {
            this.indexBuildingThread = new IndexBuildingThread(this.indexDirectory);
        }
        try {
            this.indexBuildingThread.start();
        }
        catch (IllegalThreadStateException itse) {
            log.debug((Object)"Ignoring request to build new index. Already in progress.");
        }
    }

    static {
        BooleanQuery.setMaxClauseCount((int)Integer.MAX_VALUE);
    }

    private class ScheduledIndexingTimerTask
    extends TimerTask {
        private ScheduledIndexingTimerTask() {
        }

        public void run() {
            Date nextExecutionTime = new Date(this.scheduledExecutionTime() + (long)AutorebuildingDirectoryIndex.this.indexingSchedulePeriodInMilliseconds);
            log.info((Object)("Starting scheduled index rebuild. Next rebuild at " + AutorebuildingDirectoryIndex.this.formatDatetime(nextExecutionTime)));
            AutorebuildingDirectoryIndex.this.rebuild();
        }
    }
}

