/*
 * Decompiled with CFR 0.152.
 */
package imcode.server.document.index;

import imcode.server.document.DocumentDomainObject;
import imcode.server.document.index.DirectoryIndex;
import imcode.server.document.index.DocumentIndex;
import imcode.server.document.index.IndexException;
import imcode.server.user.UserDomainObject;
import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import org.apache.commons.io.FileUtils;
import org.apache.log4j.Logger;
import org.apache.log4j.NDC;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Query;

public class AutorebuildingDirectoryIndex
implements DocumentIndex {
    private int indexingSchedulePeriodInMilliseconds;
    private static final Logger log = Logger.getLogger((String)AutorebuildingDirectoryIndex.class.getName());
    private DirectoryIndex index;
    private Set documentsToAddToNewIndex = Collections.synchronizedSet(new HashSet());
    private Set documentsToRemoveFromNewIndex = Collections.synchronizedSet(new HashSet());
    private final Object newIndexBuildingLock = new Object();
    private boolean buildingNewIndex;

    public AutorebuildingDirectoryIndex(File indexDirectory, int indexingSchedulePeriodInMinutes) {
        this.indexingSchedulePeriodInMilliseconds = indexingSchedulePeriodInMinutes * 60000;
        this.index = new DirectoryIndex(indexDirectory);
        Timer scheduledIndexBuildingTimer = new Timer(true);
        long scheduledIndexDelay = 0L;
        if (IndexReader.indexExists((File)indexDirectory)) {
            try {
                long indexModifiedTime = IndexReader.lastModified((File)indexDirectory);
                long time = System.currentTimeMillis();
                long nextTime = indexModifiedTime + (long)this.indexingSchedulePeriodInMilliseconds;
                if (nextTime > time) {
                    log.info((Object)("First indexing scheduled at " + this.formatDatetime(new Date(nextTime))));
                    scheduledIndexDelay = nextTime - time;
                }
            }
            catch (IOException e) {
                log.warn((Object)"Failed to get last modified time of index.", (Throwable)e);
            }
        }
        scheduledIndexBuildingTimer.scheduleAtFixedRate((TimerTask)new ScheduledIndexingTimerTask(), scheduledIndexDelay, (long)this.indexingSchedulePeriodInMilliseconds);
    }

    private String formatDatetime(Date nextExecutionTime) {
        return new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(nextExecutionTime);
    }

    public synchronized void indexDocument(DocumentDomainObject document) {
        if (this.buildingNewIndex) {
            this.documentsToAddToNewIndex.add(document);
        }
        try {
            this.index.indexDocument(document);
        }
        catch (IndexException e) {
            log.error((Object)("Failed to index document " + document.getId() + ". Reindexing..."));
            this.buildNewIndexInBackground();
        }
    }

    public synchronized void removeDocument(DocumentDomainObject document) {
        if (this.buildingNewIndex) {
            this.documentsToRemoveFromNewIndex.add(document);
        }
        try {
            this.index.removeDocument(document);
        }
        catch (IndexException e) {
            log.error((Object)("Failed to remove document " + document.getId() + " from index. Reindexing..."));
            this.buildNewIndexInBackground();
        }
    }

    public synchronized DocumentDomainObject[] search(Query query, UserDomainObject searchingUser) throws IndexException {
        try {
            DocumentDomainObject[] documents = this.index.search(query, searchingUser);
            if (this.index.isInconsistent()) {
                this.buildNewIndexInBackground();
            }
            return documents;
        }
        catch (IndexException ex) {
            log.warn((Object)"Search failed", (Throwable)ex);
            this.buildNewIndexInBackground();
            throw ex;
        }
    }

    public void rebuild() {
        this.buildNewIndexInBackground();
    }

    private void buildNewIndexInBackground() {
        Thread indexBuildingThread = new Thread("Background indexing thread"){

            public void run() {
                AutorebuildingDirectoryIndex.this.buildNewIndex();
            }
        };
        int callersPriority = Thread.currentThread().getPriority();
        int newPriority = Math.max(callersPriority - 1, 1);
        indexBuildingThread.setPriority(newPriority);
        indexBuildingThread.setDaemon(true);
        indexBuildingThread.start();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void buildNewIndex() {
        NDC.push((String)"buildNewIndex");
        try {
            File indexDirectoryFile = this.index.getDirectory();
            File parentFile = indexDirectoryFile.getParentFile();
            String name = indexDirectoryFile.getName();
            this.buildNewIndex(parentFile, name);
        }
        catch (IOException e) {
            log.fatal((Object)"Failed to index all documents.", (Throwable)e);
        }
        finally {
            NDC.pop();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void buildNewIndex(File parentFile, String name) throws IOException {
        if (this.buildingNewIndex) {
            log.debug((Object)"Ignoring request to build new index. Already in progress.");
            return;
        }
        Object object = this.newIndexBuildingLock;
        synchronized (object) {
            this.buildingNewIndex = true;
            File newIndexDirectoryFile = new File(parentFile, name + ".new");
            DirectoryIndex newIndexDirectory = new DirectoryIndex(newIndexDirectoryFile);
            newIndexDirectory.indexAllDocuments();
            this.replaceIndexWithNewIndex(newIndexDirectory);
            this.buildingNewIndex = false;
        }
        this.considerDocumentsForNewIndex();
    }

    private synchronized void considerDocumentsForNewIndex() throws IndexException {
        DocumentDomainObject document;
        Iterator iterator = this.documentsToAddToNewIndex.iterator();
        while (iterator.hasNext()) {
            document = (DocumentDomainObject)iterator.next();
            this.index.indexDocument(document);
            iterator.remove();
        }
        iterator = this.documentsToRemoveFromNewIndex.iterator();
        while (iterator.hasNext()) {
            document = (DocumentDomainObject)iterator.next();
            this.index.removeDocument(document);
            iterator.remove();
        }
    }

    private synchronized void replaceIndexWithNewIndex(DirectoryIndex newIndex) throws IOException {
        File newIndexDirectory;
        File indexDirectory = this.index.getDirectory();
        File oldIndex = new File(indexDirectory.getParentFile(), indexDirectory.getName() + ".old");
        if (oldIndex.exists()) {
            FileUtils.forceDelete((File)oldIndex);
        }
        if (indexDirectory.exists() && !indexDirectory.renameTo(oldIndex)) {
            log.error((Object)("Failed to rename \"" + indexDirectory + "\" to \"" + oldIndex + "\"."));
        }
        if (!(newIndexDirectory = newIndex.getDirectory()).renameTo(indexDirectory)) {
            throw new IOException("Failed to rename \"" + newIndexDirectory + "\" to \"" + indexDirectory + "\".");
        }
        FileUtils.deleteDirectory((File)oldIndex);
    }

    static {
        BooleanQuery.setMaxClauseCount((int)Integer.MAX_VALUE);
    }

    private class ScheduledIndexingTimerTask
    extends TimerTask {
        private ScheduledIndexingTimerTask() {
        }

        public void run() {
            Date nextExecutionTime = new Date(this.scheduledExecutionTime() + (long)AutorebuildingDirectoryIndex.this.indexingSchedulePeriodInMilliseconds);
            log.info((Object)("Starting scheduled indexing. Next indexing at " + AutorebuildingDirectoryIndex.this.formatDatetime(nextExecutionTime)));
            AutorebuildingDirectoryIndex.this.buildNewIndexInBackground();
        }
    }
}

