package com.imcode.util;

import org.apache.commons.collections.MultiHashMap;
import org.apache.commons.collections.MultiMap;
import org.apache.commons.collections.iterators.IteratorEnumeration;
import org.apache.commons.fileupload.*;
import org.apache.commons.lang.UnhandledException;
import org.apache.oro.text.perl.Perl5Util;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import javax.activation.DataSource;
import java.io.*;
import java.util.*;

/**
 * Extends HttpServletRequest for RFC 1867 form based file upload support from Jakarta Commons-FileUpload
 */
public class MultipartHttpServletRequest extends HttpServletRequestWrapper {
    static final String CHARSET_ISO_8859_1 = "ISO-8859-1";

    private MultiMap fileItemMap;

    public MultipartHttpServletRequest( HttpServletRequest request ) throws IOException {
        this(request,-1) ;
    }

    /**
        Constructor used to specify a size limit for uploads.

        @param sizeLimit The maximum size for an upload, in bytes.
     **/
    public MultipartHttpServletRequest( HttpServletRequest request, long sizeLimit ) throws IOException {
        super( request );
        if ( FileUploadBase.isMultipartContent( request ) ) {
            FileUpload fileUpload = new FileUpload( new DefaultFileItemFactory() );
            fileUpload.setSizeMax( sizeLimit );
            List fileItems ;
            try {
                fileItems = fileUpload.parseRequest( request );
            } catch ( FileUploadException e ) {
                throw new IOException( e.getMessage() );
            }
            fileItemMap = new MultiHashMap();
            for ( Iterator iterator = fileItems.iterator(); iterator.hasNext(); ) {
                FileItem fileItem = new BaseNameFileItem( (FileItem)iterator.next() );
                fileItemMap.put( fileItem.getFieldName(), fileItem );
            }
        }
    }

    public String getParameter( String key ) {
        String[] parameterValues = getParameterValues( key );
        if ( null != parameterValues && parameterValues.length > 0 ) {
            return parameterValues[0];
        }
        return null;
    }

    public DataSourceFileItem getParameterFileItem( String key ) {
        DataSourceFileItem[] parameterValues = getParameterFileItems( key );
        if ( null != parameterValues && parameterValues.length > 0 ) {
            return parameterValues[0];
        }
        return null;
    }

    public DataSourceFileItem[] getParameterFileItems( String key ) {
        if ( null == fileItemMap ) {
            return null;
        }
        final Collection parameterFileItems = (Collection)fileItemMap.get( key );
        if ( null == parameterFileItems ) {
            return null;
        }
        return (DataSourceFileItem[])parameterFileItems.toArray( new DataSourceFileItem[parameterFileItems.size()] );
    }

    public Map getParameterMap() {
        Map map = super.getParameterMap() ;
        if ( null != fileItemMap ) {
            map = new HashMap( map );
            Set fileItemKeys = fileItemMap.keySet();
            for ( Iterator iterator = fileItemKeys.iterator(); iterator.hasNext(); ) {
                String key = (String)iterator.next();
                map.put( key, getParameterValues( key ) );
            }
        }
        return map;
    }

    public Enumeration getParameterNames() {
        Enumeration superParameterNames = super.getParameterNames();
        Set parameterNames = new HashSet();
        while ( superParameterNames.hasMoreElements() ) {
            parameterNames.add( superParameterNames.nextElement() );
        }
        if (null != fileItemMap) {
            parameterNames.addAll( fileItemMap.keySet() );
        }
        return new IteratorEnumeration( parameterNames.iterator() );
    }

    public String[] getParameterValues( String key ) {
        if ( null != fileItemMap ) {
            List parameterValues = new ArrayList();
            Collection fileItems = (Collection)fileItemMap.get( key );
            if ( null == fileItems ) {
                return null;
            }
            for ( Iterator iterator = fileItems.iterator(); iterator.hasNext(); ) {
                FileItem fileItem = (FileItem)iterator.next();
                String contentType = fileItem.getContentType();
                parameterValues.add( getStringFromBytesWithContentType( fileItem.get(), contentType ) );
            }
            return (String[])parameterValues.toArray( new String[parameterValues.size()] );
        }
        return super.getParameterValues( key );
    }

    static String getStringFromBytesWithContentType( byte[] bytes, String contentType ) {
        try {
            return new String( bytes, getCharsetFromContentType(contentType) );
        } catch ( UnsupportedEncodingException ignored ) {
        }
        try {
            return new String( bytes, CHARSET_ISO_8859_1 );
        } catch ( UnsupportedEncodingException never ) {
            throw new UnhandledException(never);
        }
    }

    static String getCharsetFromContentType(String contentType) {
        String charset = CHARSET_ISO_8859_1;
        if ( null != contentType ) {
            Perl5Util perl5Util = new Perl5Util();
            if ( perl5Util.match( "/\\bcharset=\"?(\\S+?)\"?(?:$|;|\\s)/", contentType ) ) {
                charset = perl5Util.group( 1 );
            }
        }
        return charset ;
    }

    public interface DataSourceFileItem extends FileItem, DataSource {

    }

    private class BaseNameFileItem extends FileItemWrapper {

        private BaseNameFileItem( FileItem fileItem ) {
            super( fileItem );
        }

        public String getName() {
            String filename = fileItem.getName();
            if ( null != filename ) {
                filename = filename.substring( filename.lastIndexOf( '/' ) + 1 );
                filename = filename.substring( filename.lastIndexOf( '\\' ) + 1 );
            }
            return filename;
        }
    }

    private class FileItemWrapper implements DataSourceFileItem {

        FileItem fileItem;

        private FileItemWrapper( FileItem fileItem ) {
            this.fileItem = fileItem;
        }

        public String getContentType() {
            return fileItem.getContentType();
        }

        public String getFieldName() {
            return fileItem.getFieldName();
        }

        public void setFieldName( String s ) {
            fileItem.setFieldName( s );
        }

        public InputStream getInputStream() throws IOException {
            return fileItem.getInputStream();
        }

        public String getName() {
            return fileItem.getName();
        }

        public OutputStream getOutputStream() throws IOException {
            return fileItem.getOutputStream();
        }

        public long getSize() {
            return fileItem.getSize();
        }

        public String getString() {
            return fileItem.getString();
        }

        public boolean isFormField() {
            return fileItem.isFormField();
        }

        public void setFormField( boolean b ) {
            fileItem.setFormField( b );
        }

        public boolean isInMemory() {
            return fileItem.isInMemory();
        }

        public byte[] get() {
            return fileItem.get();
        }

        public void delete() {
            fileItem.delete();
        }

        public String getString( String s ) throws UnsupportedEncodingException {
            return fileItem.getString( s );
        }

        public void write( File file ) throws Exception {
            fileItem.write( file );
        }
    }

}